/*
 * Copyright (C) 2001-2011 Jacek Sieka, arnetheduck on gmail point com
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/*
 * Automatic Directory Listing Search
 * Henrik Engström, henrikengstrom at home se
 */

#ifndef DCPLUSPLUS_DCPP_A_D_L_SEARCH_H
#define DCPLUSPLUS_DCPP_A_D_L_SEARCH_H

#include <boost/regex.hpp>
#include <boost/variant.hpp>

#include "Util.h"

#include "SettingsManager.h"
#include "ResourceManager.h"
#include "FavoriteManager.h"

#include "StringSearch.h"
#include "Singleton.h"
#include "DirectoryListing.h"

namespace dcpp {

class AdlSearchManager;

///	Class that represent an ADL search
class ADLSearch
{
public:
	ADLSearch();

	/// The search string
	string searchString;									 

	/// Active search
	bool isActive;
	
	/// Forbidden file
	bool isForbidden;
	// Raw command
	int raw;
	// Casesensitive
	bool isCaseSensitive;
	// Some Comment
	string adlsComment;
	// Global or not
	bool isGlobal;
	// Priority
	int adlsPriority;

	/// Auto Queue Results
	bool isAutoQueue;

	/// Search source type
	enum SourceType {
		TypeFirst = 0,
		OnlyFile = TypeFirst,
		OnlyDirectory,
		FullPath,
		TTHash,
		TypeLast
	} sourceType;

	SourceType StringToSourceType(const string& s);
	string SourceTypeToString(SourceType t);
	tstring SourceTypeToDisplayString(SourceType t);

	string getConfStr() {
		string ConfStr;
		if(isGlobal)
			ConfStr += STRING(GLOBAL);
		if(isRegEx())
			ConfStr = ConfStr + (isGlobal ? ", " : "") + (isCaseSensitive ? STRING(REGEXP) : STRING(CASE_INSENSITIVE_REGEXP));
		return (!ConfStr.empty() ? " (" + ConfStr + ")" : ConfStr);
	}

	// Maximum & minimum file sizes (in bytes). 
	// Negative values means do not check.
	int64_t minFileSize;
	int64_t maxFileSize;

	enum SizeType {
		SizeBytes     = TypeFirst,
		SizeKiloBytes,
		SizeMegaBytes,
		SizeGigaBytes
	};

	SizeType typeFileSize;

	SizeType StringToSizeType(const string& s);
	string SizeTypeToString(SizeType t);
	static tstring SizeTypeToDisplayString(SizeType t);
	int64_t GetSizeBase();

	/// Name of the destination directory (empty = 'ADLSearch') and its index
	string destDir;
	unsigned long ddIndex;

	bool isRegEx() const;
	void setRegEx(bool b);

private:
	friend class ADLSearchManager;

	boost::variant<StringSearch::List, boost::regex> v;

	/// Prepare search
	void prepare(StringMap& params);

	/// Search for file match 
	bool matchesFile(const string& f, const string& fp, const string& root, int64_t size, bool noAdlSearch);
	/// Search for directory match 
	bool matchesDirectory(const string& d, bool noAdlSearch);

	bool searchAll(const string& s);
};

///	Class that holds all active searches
class ADLSearchManager : public Singleton<ADLSearchManager>
{
public:
	// Destination directory indexing
	struct DestDir {
		string name;
		DirectoryListing::Directory* dir;
		DirectoryListing::Directory* subdir;
		bool fileAdded;
		DestDir() : name(""), dir(NULL), subdir(NULL) {}
	};
	typedef vector<DestDir> DestDirList;

	ADLSearchManager();
	~ADLSearchManager();

	// Search collection
	typedef vector<ADLSearch> SearchCollection;
	SearchCollection collection;

	// Load/save search collection to XML file
	void load();
	void save();

	// Settings
	GETSET(bool, breakOnFirst, BreakOnFirst);
	GETSET(HintedUser, user, User);
	GETSET(bool, noAdlSearch, NoAdlSearch);

	// @remarks Used to add ADLSearch directories to an existing DirectoryListing
	void matchListing(DirectoryListing& aDirList);

private:
	// @internal
	void matchRecurse(DestDirList& aDestList, DirectoryListing& filelist, DirectoryListing::Directory* aDir, string& aPath);
	// Search for file match
	void matchesFile(DestDirList& destDirVector, DirectoryListing::File *currentFile, string& fullPath);
	// Search for directory match
	void matchesDirectory(DestDirList& destDirVector, DirectoryListing::Directory* currentDir, string& fullPath);
	// Step up directory
	void stepUpDirectory(DestDirList& destDirVector);

	// Prepare destination directory indexing
	void prepareDestinationDirectories(DestDirList& destDirVector, DirectoryListing::Directory* root, StringMap& params);
	// Finalize destination directories
	void finalizeDestinationDirectories(DestDirList& destDirVector, DirectoryListing::Directory* root);

	static string getConfigFile();
};

} // namespace dcpp

#endif // !defined(DCPLUSPLUS_DCPP_A_D_L_SEARCH_H)


/**
 * @file
 * $Id$
  */
