/*

PROJECTION MATRIX
=================

Matrices in GC are mirrored. Here are matrices described that are GL compatible.
Matrices in GL convert to linear by columns. 

XF_1020	PROJ_MTX_0		P0
XF_1021	PROJ_MTX_1		P1
XF_1022	PROJ_MTX_2		P2
XF_1023	PROJ_MTX_3		P3
XF_1024	PROJ_MTX_4		P4
XF_1025	PROJ_MTX_5		P5
XF_1026	PROJ_MTX_MODE	0-PERSPECTVE 1-ORTHOGRAPHIC

PERSPECTIVE PROJECTION (0)

P0  0 P1  0
 0 P2 P3  0
 0  0 P4 P5
 0  0 -1  0

ORTHOGRPHIC PROJECTION (1)

P0  0  0 P1
 0 P2  0 P3
 0  0 P4 P5
 0  0  0  1

*** IMPORTANT ***

Matrix Frustum on GC works different way than on PC. Calculation returns different values for P4 and P5.
It needs to be verified if conversion is required to PC values.

------------------------------------------------------------------------------------------------------------

MODELVIEW MATRIX
================

Matrices are stored at the beginning of XF Regs
Matrix is in row order 4x3 without last row which should be 0,0,0,1 for OpenGL


*/

#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <windows.h>


#include <GL/glew.h>

#if WITH_CG
#include <Cg/cg.h>
#include <Cg/cgGL.h>
#endif

#include "gp_defs.h"
#include "xf_regs.h"
#include "graphics_processor/vertex_processor.h"
#include "debug/tracers.h"

gp_reg	gp_xf_regs[0x3000];

float	xf_modelview_matrix[16];
float	xf_normal_matrix[16];
uint32	xf_modelview_matrix_addr = 0;
uint32	xf_normal_matrix_addr = 0x400;
GLfloat	xf_projection_matrix[16];
GLfloat xf_texture_matrix[8][16];
uint32	xf_texture_matrix_addr[8];

#if WITH_VERTEX_SHADER
#if WITH_CG
extern CGparameter vs_projection_mtx;
extern CGparameter vs_modelview_mtx;
extern CGparameter vs_texture_mtx[];
extern CGparameter vs_normal_mtx;
extern CGparameter	light_enabled[8];
extern CGparameter	lighting;
#else
#error Currently Vertex Shader allowed only through CG
#endif // WITH CG
#endif // WITH_VERTEX_SHADER

float xf_one[4] = 
{ 1.0f, 1.0f, 1.0f, 1.0f};
float xf_zero[4] = 
{ 0.0f, 0.0f, 0.0f, 0.0f};

void xf_set_projection_matrix(uint32 data)
{
	if (data == 0)
	{
/*		float     left    = 24.0F;
		float     top     = 32.0F;
		float     znear   = 50.0F;
		float     zfar    = 2000.0F;
		// PERSPECTIVE MATRIX
		glMatrixMode(GL_PROJECTION);
		glLoadIdentity();
		glFrustum(-top, top, -left, left, znear, zfar);
		printf("GP: Loaded projection matrix\n");
*/		
		int i;

		for(i = 0 ; i < 16 ; i++)
			xf_projection_matrix[i] = 0.0f;

		xf_projection_matrix[0] = gp_xf_regs[0x1020].f;
		xf_projection_matrix[8] = gp_xf_regs[0x1021].f;
		xf_projection_matrix[5] = gp_xf_regs[0x1022].f;
		xf_projection_matrix[9] = gp_xf_regs[0x1023].f;
		xf_projection_matrix[10] = gp_xf_regs[0x1024].f;
		xf_projection_matrix[14] = gp_xf_regs[0x1025].f;
		xf_projection_matrix[11] = -1.0f;

#if 0 //WITH_VERTEX_SHADER
#if WITH_CG
		cgGLSetMatrixParameterfc(vs_projection_mtx, xf_projection_matrix);
#else	// WITH_CG
#error Currently Vertex Shader allowed only through CG
#endif	// WITH CG
#else	// WITH_VERTEX_SHADER

		float w1 = xf_projection_matrix[10];
		float w2 = xf_projection_matrix[14];
		float org_far = w2/w1;
		float org_near = w1 * org_far / (w1 - 1.0f);

		w1 = xf_projection_matrix[0];
		w2 = xf_projection_matrix[8];
		float org_right = ((2 * org_near / w1)*(2 * org_near / w1) - w2) / ( 2 * 2 * org_near / w1);
		float org_left  = org_right - 2*org_near/w1;
		
		w1 = xf_projection_matrix[5];
		w2 = xf_projection_matrix[9];
		float org_top = ((2 * org_near / w1)*(2 * org_near / w1) - w2) / ( 2 * 2 * org_near / w1);
		float org_bottom  = org_top - 2*org_near/w1;
/*
		printf("FAR: %f %f %f\n", w2, w1, w2/w1);
		printf("NEAR: %f\n", org_near);
		printf("RIGHT: %f\n", org_right);
		printf("LEFT: %f\n", org_left);
		printf("TOP: %f\n", org_top);
		printf("BOTTOM: %f\n", org_bottom);

		printf("GP: Projection Matrix\n");
*/
		glMatrixMode(GL_PROJECTION);
		glLoadIdentity();
		glFrustum(org_left, org_right, org_bottom, org_top, org_near, org_far);
//		glGetFloatv(GL_PROJECTION, xf_projection_matrix);
#endif // WITH_VERTEX_SHADER
	}
	else
	{
		// 1 - ORTHOGRAPHIC MATRIX
		int i;

		for(i = 0 ; i < 16 ; i++)
			xf_projection_matrix[i] = 0.0f;
		
		xf_projection_matrix[0] = gp_xf_regs[0x1020].f;
		xf_projection_matrix[12] = gp_xf_regs[0x1021].f;
		xf_projection_matrix[5] = gp_xf_regs[0x1022].f;
		xf_projection_matrix[13] = gp_xf_regs[0x1023].f;
		xf_projection_matrix[10] = gp_xf_regs[0x1024].f;
		xf_projection_matrix[14] = gp_xf_regs[0x1025].f;
		xf_projection_matrix[15] = 1.0f;
		// conversion should be here for [14]

#if 0//WITH_VERTEX_SHADER
#if WITH_CG
		cgGLSetMatrixParameterfc(vs_projection_mtx, xf_projection_matrix);
#else // WITH_CG
#error Currently Vertex Shader allowed only through CG
#endif // WITH CG
#else // WITH_VERTEX_SHADER
		float w1 = xf_projection_matrix[10];
		float w2 = xf_projection_matrix[14];

		float org_far = (w2 - 1.0f)/(2*w1);
		float org_near = org_far + 1.0f/w1;

		w1 = xf_projection_matrix[0];
		w2 = xf_projection_matrix[12];

		float org_right = (-w2 + 1.0f)/w1;
		float org_left = org_right - 2.0f/w1;

		w1 = xf_projection_matrix[5];
		w2 = xf_projection_matrix[13];

		float org_top = (-w2 + 1.0f) / w1;
		float org_bottom = org_top - 2.0f/w1;
/*
		printf("FAR: %f %f %f\n", w2, w1, w2/w1);
		printf("NEAR: %f\n", org_near);
		printf("RIGHT: %f\n", org_right);
		printf("LEFT: %f\n", org_left);
		printf("TOP: %f\n", org_top);
		printf("BOTTOM: %f\n", org_bottom);
*/
		glMatrixMode(GL_PROJECTION);
		glLoadIdentity();
		glOrtho(org_left, org_right, org_bottom, org_top, org_near, org_far);
#endif // WITH_VERTEX_SHADER
	}
}

void xf_set_modelview_matrix(uint32 addr)
{
	int i;
	uint32 crc = 0;
	static uint32 oldcrc = 0;
	for(i = 0 ; i < 12 ; i++)
	{
		crc += gp_xf_regs[addr + i].i;
	}

	if (crc == oldcrc)
		return;
	oldcrc = crc;
	xf_modelview_matrix[0] = gp_xf_regs[addr + 0].f;
	xf_modelview_matrix[1] = gp_xf_regs[addr + 4].f;
	xf_modelview_matrix[2] = gp_xf_regs[addr + 8].f;
	xf_modelview_matrix[3] = 0.0f;
	xf_modelview_matrix[4] = gp_xf_regs[addr + 1].f;
	xf_modelview_matrix[5] = gp_xf_regs[addr + 5].f;
	xf_modelview_matrix[6] = gp_xf_regs[addr + 9].f;
	xf_modelview_matrix[7] = 0.0f;
	xf_modelview_matrix[8] = gp_xf_regs[addr + 2].f;
	xf_modelview_matrix[9] = gp_xf_regs[addr + 6].f;
	xf_modelview_matrix[10] = gp_xf_regs[addr + 10].f;
	xf_modelview_matrix[11] = 0.0f;
	xf_modelview_matrix[12] = gp_xf_regs[addr + 3].f;
	xf_modelview_matrix[13] = gp_xf_regs[addr + 7].f;
	xf_modelview_matrix[14] = gp_xf_regs[addr + 11].f;
	xf_modelview_matrix[15] = 1.0f;

//	printf("GP: Loaded modelview matrix\n");

#if WITH_VERTEX_SHADER
#if WITH_CG
	// maybe replace with direct matrix params
	glMatrixMode(GL_MODELVIEW);
	glLoadMatrixf(xf_modelview_matrix);
//	cgGLSetMatrixParameterfc(vs_modelview_mtx, xf_modelview_matrix);
#else	// WITH_CG
#error Currently Vertex Shader allowed only through CG
#endif	// WITH CG
#else
	glMatrixMode(GL_MODELVIEW);
	glLoadMatrixf(xf_modelview_matrix);
#endif // WITH_VERTEX_SHADER
//	cgGLSetMatrixParameterfr(vs_modelview_mtx, &gp_xf_regs[addr].f);
}

void xf_set_normal_matrix(uint32 addr)
{
	// OpenGL does not let setting Normal Matrix
	// it is calculated as Transpose Inverse of Modelview matrix internally in OpenGL
#if WITH_VERTEX_SHADER
#if WITH_CG
	cgGLSetMatrixParameterfr(vs_normal_mtx, &gp_xf_regs[addr].f);
#else	// WITH_CG
#error Currently Vertex Shader allowed only through CG
#endif	// WITH CG
#endif // WITH_VERTEX_SHADER
}

void xf_set_texture_matrix(uint32 id)
{
#if WITH_VERTEX_SHADER
#if WITH_CG
	uint32 addr;
	addr = xf_texture_matrix_addr[id];
	cgGLSetMatrixParameterfr(vs_texture_mtx[id], &gp_xf_regs[addr].f);
//	cgGLSetMatrixParameterfc(vs_texture_mtx[id], xf_texture_matrix[id]);
#else	// WITH_CG
#error Currently Vertex Shader allowed only through CG
#endif	// WITH CG
#else
	int i;
	uint32 addr;
	addr = xf_texture_matrix_addr[id];

	for(i = 0 ; i < 16 ; i++)
		xf_texture_matrix[id][i] = 0.0f;

	xf_texture_matrix[id][0] = gp_xf_regs[addr + 0].f;
	xf_texture_matrix[id][1] = gp_xf_regs[addr + 4].f;
	xf_texture_matrix[id][2] = gp_xf_regs[addr + 8].f;
	xf_texture_matrix[id][3] = 0.0f;

	xf_texture_matrix[id][4] = gp_xf_regs[addr + 1].f;
	xf_texture_matrix[id][5] = gp_xf_regs[addr + 5].f;
	xf_texture_matrix[id][6] = gp_xf_regs[addr + 9].f;
	xf_texture_matrix[id][7] = 0.0f;

	xf_texture_matrix[id][8] = gp_xf_regs[addr + 2].f;
	xf_texture_matrix[id][9] = gp_xf_regs[addr + 6].f;
	xf_texture_matrix[id][10] = gp_xf_regs[addr + 10].f;
	xf_texture_matrix[id][11] = 0.0f;

	xf_texture_matrix[id][12] = gp_xf_regs[addr + 3].f;
	xf_texture_matrix[id][13] = gp_xf_regs[addr + 7].f;
	xf_texture_matrix[id][14] = gp_xf_regs[addr + 11].f;
	xf_texture_matrix[id][15] = 1.0f;

//	syslog(XF,"Load texture matrix %x\n", addr);
	glMatrixMode(GL_TEXTURE);
	glLoadMatrixf(xf_texture_matrix[id]);
	glMatrixMode(GL_MODELVIEW);

#endif // WITH_VERTEX_SHADER
}


typedef struct
{
	vx_vertex_t pos[4];
	vx_vertex_t dir[4];
	GLfloat		col[4];
} xf_light_t;

GLfloat mat_col[4] = { 1.5, 0.5, 0.5, 1.0 };
GLfloat col0_amb[4];

xf_light_t	xf_light[8];


void xf_update_chanctrl(uint32 chan)
{
	uint32 data = gp_xf_regs[0x100e].i;

	switch((data & 0x1) | ((data >> 5) & 0x2))
	{
	case 0x3:	// AMB == 1 && DIF == 1
		syslog(XF,"COL0CTRL Enable Material Color for Ambient and Diffuse\n");
		glColorMaterial(GL_FRONT, GL_AMBIENT_AND_DIFFUSE);
		glEnable(GL_COLOR_MATERIAL);
		break;
	case 0x2:	// AMB == 1 && DIF == 0
		syslog(XF,"COL0CTRL Enable Material Color for Ambient\n");
		// diffuse comes from COL0MAT register XF_100c
		glMaterialfv(GL_FRONT, GL_DIFFUSE, mat_col);
		glColorMaterial(GL_FRONT, GL_AMBIENT);
		glEnable(GL_COLOR_MATERIAL);
		glColor4fv(col0_amb);
		break;
	case 0x1:	// AMB == 0 && DIF == 1
		syslog(XF,"COL0CTRL Enable Material Color for Diffuse\n");
		// ambient comes from COL0AMB register XF_100a
		glMaterialfv(GL_FRONT, GL_AMBIENT, col0_amb);
		glColorMaterial(GL_FRONT, GL_DIFFUSE);
		glEnable(GL_COLOR_MATERIAL);
		glColor4fv(mat_col);
		break;
	case 0x0:	// AMB == 0 && DIF == 0
		syslog(XF,"COL0CTRL Disable Material Color\n");
		// both colours come from their registers
		glEnable(GL_COLOR_MATERIAL);
		glColorMaterial(GL_FRONT_AND_BACK, GL_AMBIENT);
		glColor4fv(col0_amb);
		glColorMaterial(GL_FRONT_AND_BACK, GL_DIFFUSE);
		glColor4fv(mat_col);
		//glDisable(GL_COLOR_MATERIAL);
		glColorMaterial(GL_FRONT_AND_BACK, GL_AMBIENT_AND_DIFFUSE);
		glEnable(GL_COLOR_MATERIAL);
		//			glMaterialfv(GL_FRONT, GL_AMBIENT, col0_amb);
		//			glMateialfv(GL_FRONT, GL_DIFFUSE, mat_col);
		break;
	}


#if WITH_VERTEX_SHADER
#if WITH_CG
//#error	Lighting with CG not yet implemented correctly
	if (data & 0x2)
	{
		syslog(XF,"COL0CTRL Enable Lighting\n");
		cgGLSetParameter1fv(lighting, xf_one);
		glEnable(GL_LIGHTING);
		glEnable(GL_NORMALIZE);
	}
	else
	{
		syslog(XF,"COL0CTRL Disable Lighting\n");
		cgGLSetParameter1fv(lighting, xf_zero);
		glDisable(GL_LIGHTING);
	}

	uint32 i;
	uint32 n;
	for(i = 0 ; i < 8 ; i++)
	{
		n = (i < 4)?(i + 2):(i + 7);
		if (data & (1 << n))
		{
			syslog(XF,"COL0CTRL Enable Light %d\n", i);
			cgGLSetParameter1fv(light_enabled[i], xf_one);
		}
		else
		{
			syslog(XF,"COL0CTRL Disable Light %d\n", i);
			cgGLSetParameter1fv(light_enabled[i], xf_zero);
		}
	}

#else	// WITH_CG
#error Currently Vertex Shader allowed only through CG
#endif	// WITH CG
#else
	if (data & 0x2)
	{
		syslog(XF,"COL0CTRL Enable Lighting\n");
		glEnable(GL_LIGHTING);
		glEnable(GL_NORMALIZE);
	}
	else
	{
		syslog(XF,"COL0CTRL Disable Lighting\n");
		glDisable(GL_LIGHTING);
	}

	if (data & (1 << 2))
	{
		syslog(XF,"COL0CTRL Enable Light 0\n");
		glEnable(GL_LIGHT0);
	}
	else
	{
		syslog(XF,"COL0CTRL Disable Light 0\n");
		glDisable(GL_LIGHT0);
	}

	if (data & (1 << 3))
	{
		syslog(XF,"COL0CTRL Enable Light 1\n");
		glEnable(GL_LIGHT1);
	}
	else
	{
		syslog(XF,"COL0CTRL Disable Light 1\n");
		glDisable(GL_LIGHT1);
	}

	if (data & (1 << 4))
	{
		syslog(XF,"COL0CTRL Enable Light 2\n");
		glEnable(GL_LIGHT2);
	}
	else
	{
		syslog(XF,"COL0CTRL Disable Light 2\n");
		glDisable(GL_LIGHT2);
	}
	if (data & (1 << 5))
	{
		syslog(XF,"COL0CTRL Enable Light 3\n");
		glEnable(GL_LIGHT3);
	}
	else
	{
		syslog(XF,"COL0CTRL Disable Light 3\n");
		glDisable(GL_LIGHT3);
	}

	if (data & (1 << 11))
	{
		syslog(XF,"COL0CTRL Enable Light 4\n");
		glEnable(GL_LIGHT4);
	}
	else
	{
		syslog(XF,"COL0CTRL Disable Light 4\n");
		glDisable(GL_LIGHT4);
	}

	if (data & (1 << 12))
	{
		syslog(XF,"COL0CTRL Enable Light 5\n");
		glEnable(GL_LIGHT5);
	}
	else
	{
		syslog(XF,"COL0CTRL Disable Light 5\n");
		glDisable(GL_LIGHT5);
	}

	if (data & (1 << 13))
	{
		syslog(XF,"COL0CTRL Enable Light 6\n");
		glEnable(GL_LIGHT6);
	}
	else
	{
		syslog(XF,"COL0CTRL Disable Light 6\n");
		glDisable(GL_LIGHT6);
	}

	if (data & (1 << 14))
	{
		syslog(XF,"COL0CTRL Enable Light 7\n");
		glEnable(GL_LIGHT7);
	}
	else
	{
		syslog(XF,"COL0CTRL Disable Light 7\n");
		glDisable(GL_LIGHT7);
	}
#endif // WITH_VERTEX_SHADER
}





void gp_xf_write_reg(uint32 reg, uint32 data)
{
	gp_xf_regs[reg].i = data;

	syslog(XF,"GP: XF_%04x %08x %f\n", reg, data, gp_xf_regs[reg].f);
	// for now we want switch statement but later we need special callbacks next to registers

	if (reg >= xf_texture_matrix_addr[0] && reg <= (xf_texture_matrix_addr[0] + 8))
	{
		xf_set_texture_matrix(0);
	}
	if (reg >= xf_texture_matrix_addr[1] && reg <= (xf_texture_matrix_addr[1] + 8))
	{
		xf_set_texture_matrix(1);
	}
	if (reg >= xf_texture_matrix_addr[2] && reg <= (xf_texture_matrix_addr[2] + 8))
	{
		xf_set_texture_matrix(2);
	}
	if (reg >= xf_texture_matrix_addr[3] && reg <= (xf_texture_matrix_addr[3] + 8))
	{
		xf_set_texture_matrix(3);
	}
	if (reg >= xf_texture_matrix_addr[4] && reg <= (xf_texture_matrix_addr[4] + 8))
	{
		xf_set_texture_matrix(4);
	}
	if (reg >= xf_texture_matrix_addr[5] && reg <= (xf_texture_matrix_addr[5] + 8))
	{
		xf_set_texture_matrix(5);
	}
	if (reg >= xf_texture_matrix_addr[6] && reg <= (xf_texture_matrix_addr[6] + 8))
	{
		xf_set_texture_matrix(6);
	}
	if (reg >= xf_texture_matrix_addr[7] && reg <= (xf_texture_matrix_addr[7] + 8))
	{
		xf_set_texture_matrix(7);
	}

	if (reg >= xf_modelview_matrix_addr && reg <= (xf_modelview_matrix_addr + 11))
	{
		xf_set_modelview_matrix(xf_modelview_matrix_addr);
	}

	if (reg >= xf_normal_matrix_addr && reg <= (xf_normal_matrix_addr + 11))
	{
		xf_set_normal_matrix(xf_normal_matrix_addr);
	}

	if(reg >= 0x600 && reg < 0x680)
	{
		// light number
		uint32 id = (reg >> 4) & 0xf;
		reg &= 0xf;
		switch(reg)
		{
		case 0x3:
			// $LIGHT$ COLOUR
			xf_light[id].col[0] = ((data >> 24) & 0xff)/256.0f;
			xf_light[id].col[1] = ((data >> 16) & 0xff)/256.0f;
			xf_light[id].col[2] = ((data >> 8) & 0xff)/256.0f;
			xf_light[id].col[3] = ((data >> 0) & 0xff)/256.0f;
			glLightfv(GL_LIGHT0 + id, GL_DIFFUSE, xf_light[id].col);
			break;
		case 0xa:
		case 0xb:
		case 0xc:
			// $LIGHT$ POSITION
			xf_light[id].pos[reg - 0xa].i = data;
			xf_light[id].pos[3].f = 1.0f;
			
			syslog(XF,"glLightfv(GL_LIGHT%d, GL_POSITION, %f, %f, %f, %f)\n", id, xf_light[id].pos[0].f, xf_light[id].pos[1].f, xf_light[id].pos[2].f, xf_light[id].pos[3].f);

			glMatrixMode(GL_MODELVIEW);
			glPushMatrix();
			glLoadIdentity();
			glLightfv(GL_LIGHT0 + id, GL_POSITION, (GLfloat *)xf_light[id].pos);
			glPopMatrix();
			break;
		case 0xd:
		case 0xe:
		case 0xf:
			// $LIGHT$ DIRECTION
			xf_light[id].dir[reg - 0xd].i = data;
			xf_light[id].dir[3].f = 1.0f;
			glMatrixMode(GL_MODELVIEW);
			glPushMatrix();
			glLoadIdentity();
			//glLightfv(GL_LIGHT0 + id, GL_SPOT_DIRECTION, (GLfloat *)xf_light[id].dir);
			//glLightf(GL_LIGHT0 + id, GL_SPOT_CUTOFF, 100.0f);
			glPopMatrix();
			break;
		default:
			syslog(XF,"LIGHT: Unknown register\n");
			break;
		}
		return;
	}

	switch(reg)
	{
	case 0x100a:
		// ambient colour 1
		col0_amb[0] = ((data >> 24) & 0xff)/256.0f;
		col0_amb[1] = ((data >> 16) & 0xff)/256.0f;
		col0_amb[2] = ((data >> 8) & 0xff)/256.0f;
		col0_amb[3] = ((data >> 0) & 0xff)/256.0f;
		xf_update_chanctrl(0);
		break;
//	case 0x100b:
		// ambient colour 2
//		break;
	case 0x100c:
		// material colour 1
//		printf("MatClr %08x\n", data);
		mat_col[0] = ((data >> 24) & 0xff)/256.0f;
		mat_col[1] = ((data >> 16) & 0xff)/256.0f;
		mat_col[2] = ((data >> 8) & 0xff)/256.0f;
		mat_col[3] = ((data >> 0) & 0xff)/256.0f;
		xf_update_chanctrl(0);
		break;
//	case 0x100d:
		// material colour 2
//		break;
	case 0x100e:
		// colour 1 control
		xf_update_chanctrl(0);
		break;
	case 0x1018:
		syslog(XF,"SetMatrix: MV %x T0: %x T1: %x T2: %x T3: %x\n", data & 0x3f, (data >> 6) & 0x3f, (data >> 12) & 0x3f, (data >> 18) & 0x3f, (data >> 24) & 0x3f);
		xf_modelview_matrix_addr = (data & 0x3f) << 2;
		xf_set_modelview_matrix(xf_modelview_matrix_addr);

		xf_texture_matrix_addr[0] = (((data >> 6) & 0x3f) << 2);
		xf_texture_matrix_addr[1] = (((data >> 12) & 0x3f) << 2);
		xf_texture_matrix_addr[2] = (((data >> 18) & 0x3f) << 2);
		xf_texture_matrix_addr[3] = (((data >> 24) & 0x3f) << 2);
		xf_set_texture_matrix(0);
		xf_set_texture_matrix(1);
		xf_set_texture_matrix(2);
		xf_set_texture_matrix(3);
		break;

	case 0x1019:
		xf_texture_matrix_addr[4] = (((data >> 0) & 0x3f) << 2);
		xf_texture_matrix_addr[5] = (((data >> 6) & 0x3f) << 2);
		xf_texture_matrix_addr[6] = (((data >> 12) & 0x3f) << 2);
		xf_texture_matrix_addr[7] = (((data >> 18) & 0x3f) << 2);
		xf_set_texture_matrix(4);
		xf_set_texture_matrix(5);
		xf_set_texture_matrix(6);
		xf_set_texture_matrix(7);
		break;

	case 0x1026:
		// write to PROJ_MTX_MODE
		xf_set_projection_matrix(data);
		break;

	case 0x101f:
		{
		float x, y, w, h;
		float zfar, znear;
		w = gp_xf_regs[0x101a].f;
		h = -gp_xf_regs[0x101b].f;

		x = gp_xf_regs[0x101d].f;
		y = gp_xf_regs[0x101e].f;

		x = x - 342 - w;
		y = y - 342 - h;
		w *= 2.0f;
		h = h * 2.0f;

//		y = 480 - y;
//		h = -h;
		y = 480 - (y + h);
//		printf("org: %f, %f, %f, %f\n", gp_xf_regs[0x101a].f, gp_xf_regs[0x101b].f, gp_xf_regs[0x101d].f, gp_xf_regs[0x101e].f);
		glViewport((GLint)x, (GLint)y, (GLsizei)w, (GLsizei)h);
		syslog(XF,"glViewport(%d, %d, %d, %d)\n", (GLint)x, (GLint)y, (GLsizei)w, (GLsizei)h);
		
		zfar = gp_xf_regs[0x101f].f;
		znear = gp_xf_regs[0x101c].f - zfar;
		zfar /= 16777215.0f;
		znear /= 16777215.0f;
		glDepthRange(znear, zfar);
		syslog(XF,"Depth Range %f %f\n", znear, zfar);
		}
		break;

	case 0x1008:
		if (data & 0xf0)
		{
//			glEnable(GL_TEXTURE_2D);
		}
		else
		{
//			glDisable(GL_TEXTURE_2D);
		}
		break;

	case 0x103f:
		// texgen enabled
		// bit 0 enables texgen matrix
/*		if (data)
		{
			glTexGeni(GL_S, GL_TEXTURE_GEN_MODE, GL_SPHERE_MAP);
			glTexGeni(GL_T, GL_TEXTURE_GEN_MODE, GL_SPHERE_MAP);
			glEnable(GL_TEXTURE_GEN_S);
			glEnable(GL_TEXTURE_GEN_T);
		}
		else
		{
			glDisable(GL_TEXTURE_GEN_S);
			glDisable(GL_TEXTURE_GEN_T);
		}
*/		break;

	default:
		break;
	}
}


