/***************************************************************************
                         ogl_xf_regs.c  -  description
                             -------------------
    begin                : 
    copyright            : (C) 2005 by Duddie
    email                : duddie@walla.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version. See also the license.txt file for *
 *   additional informations.                                              *
 *                                                                         *
 ***************************************************************************/

//*************************************************************************//
// History of changes:
//
// 2005/07/10 - Pete
// - fixed and added all kind of lighting stuff, light is still badly emulated
//
// 2005/06/05 - Pete
// - fixed texture matrix
//
//*************************************************************************//

#include "stdafx.h"
#include "xf_regs.h"
#include "vertex_processor.h"
#define _IN_XFREGS
#include "externals.h"


/*

PROJECTION MATRIX
=================

Matrices in GC are mirrored. Here are matrices described that are GL compatible.
Matrices in GL convert to linear by columns. 

XF_1020 PROJ_MTX_0      P0
XF_1021 PROJ_MTX_1      P1
XF_1022 PROJ_MTX_2      P2
XF_1023 PROJ_MTX_3      P3
XF_1024 PROJ_MTX_4      P4
XF_1025 PROJ_MTX_5      P5
XF_1026 PROJ_MTX_MODE   0-PERSPECTVE 1-ORTHOGRAPHIC

PERSPECTIVE PROJECTION (0)

P0  0 P1  0
 0 P2 P3  0
 0  0 P4 P5
 0  0 -1  0

ORTHOGRPHIC PROJECTION (1)

P0  0  0 P1
 0 P2  0 P3
 0  0 P4 P5
 0  0  0  1

*** IMPORTANT ***

Matrix Frustum on GC works different way than on PC. 
Calculation returns different values for P4 and P5.
It needs to be verified if conversion is required to PC values.

------------------------------------------------------------------------------------------------------------

MODELVIEW MATRIX
================

Matrices are stored at the beginning of XF Regs
Matrix is in row order 4x3 without last row which should be 0,0,0,1 for OpenGL
*/

///////////////////////////////////////////////////////////////////////////
// GLOBALS

gp_reg  gp_xf_regs[0x3000];

float   xf_modelview_matrix[16];
float   xf_normal_matrix[16];
uint32  xf_modelview_matrix_addr = 0;
uint32  xf_normal_matrix_addr = 0x400;
GLfloat xf_projection_matrix[16];
GLfloat xf_texture_matrix[8][16];
uint32  xf_texture_matrix_addr[8];

float xf_one[4]  = { 1.0f, 1.0f, 1.0f, 1.0f};
float xf_zero[4] = { 0.0f, 0.0f, 0.0f, 0.0f};

typedef struct
{
    vx_vertex_t pos[4];
    vx_vertex_t dir[4];
    GLfloat     col[4];
    vx_vertex_t atten_cos[3];
    vx_vertex_t atten_dis[3];

} xf_light_t;

GLfloat mat_col[4] = { 1.5, 0.5, 0.5, 1.0 };
GLfloat col0_amb[4];

xf_light_t  xf_light[8];

GLfloat curr_light_attn[8][3]=
{
 1231.0f,0.0f,0.0f,
 1231.0f,0.0f,0.0f,
 1231.0f,0.0f,0.0f,
 1231.0f,0.0f,0.0f,
 1231.0f,0.0f,0.0f,
 1231.0f,0.0f,0.0f,
 1231.0f,0.0f,0.0f,
 1231.0f,0.0f,0.0f
};

GLfloat curr_light_color[8][4];

///////////////////////////////////////////////////////////////////////////
// CODE

void xf_set_projection_matrix(uint32 data)
{
 float w1,w2,org_far,org_near,org_right,org_left,org_top,org_bottom;
 int i;

 if (data == 0)
  {
   for(i = 0 ; i < 16 ; i++)
    xf_projection_matrix[i] = 0.0f;

   xf_projection_matrix[0] = gp_xf_regs[0x1020].f;
   xf_projection_matrix[8] = gp_xf_regs[0x1021].f;
   xf_projection_matrix[5] = gp_xf_regs[0x1022].f;
   xf_projection_matrix[9] = gp_xf_regs[0x1023].f;
   xf_projection_matrix[10] = gp_xf_regs[0x1024].f;
   xf_projection_matrix[14] = gp_xf_regs[0x1025].f;
   xf_projection_matrix[11] = -1.0f;

   w1 = xf_projection_matrix[10];
   w2 = xf_projection_matrix[14];
   org_far  = w2/w1;
   org_near = w1 * org_far / (w1 - 1.0f);

   w1 = xf_projection_matrix[0];
   w2 = xf_projection_matrix[8];
   org_right = ((2 * org_near / w1)*(2 * org_near / w1) - w2) / ( 2 * 2 * org_near / w1);
   org_left  = org_right - 2*org_near/w1;
        
   w1 = xf_projection_matrix[5];
   w2 = xf_projection_matrix[9];
   org_top     = ((2 * org_near / w1)*(2 * org_near / w1) - w2) / ( 2 * 2 * org_near / w1);
   org_bottom  = org_top - 2*org_near/w1;

#ifdef AUX_ILLEGALCALLS
   if(vx_started) auxprintf("illegal MatrixMode (a)!\n");
#endif

   glMatrixMode(GL_PROJECTION);
   glLoadIdentity();
   glFrustum(org_left, org_right, org_bottom, org_top, org_near, org_far);
  }
 else
  {
   // 1 - ORTHOGRAPHIC MATRIX

   for(i = 0 ; i < 16 ; i++)
    xf_projection_matrix[i] = 0.0f;
        
   xf_projection_matrix[0] = gp_xf_regs[0x1020].f;
   xf_projection_matrix[12] = gp_xf_regs[0x1021].f;
   xf_projection_matrix[5] = gp_xf_regs[0x1022].f;
   xf_projection_matrix[13] = gp_xf_regs[0x1023].f;
   xf_projection_matrix[10] = gp_xf_regs[0x1024].f;
   xf_projection_matrix[14] = gp_xf_regs[0x1025].f;
   xf_projection_matrix[15] = 1.0f;
   // conversion should be here for [14]

   w1 = xf_projection_matrix[10];
   w2 = xf_projection_matrix[14];

   org_far = (w2 - 1.0f)/(2*w1);
   org_near = org_far + 1.0f/w1;

   w1 = xf_projection_matrix[0];
   w2 = xf_projection_matrix[12];

   org_right = (-w2 + 1.0f)/w1;
   org_left = org_right - 2.0f/w1;

   w1 = xf_projection_matrix[5];
   w2 = xf_projection_matrix[13];

   org_top = (-w2 + 1.0f) / w1;
   org_bottom = org_top - 2.0f/w1;

#ifdef AUX_ILLEGALCALLS
   if(vx_started) auxprintf("illegal MatrixMode (b)!\n");
#endif

   glMatrixMode(GL_PROJECTION);
   glLoadIdentity();
   glOrtho(org_left, org_right, org_bottom, org_top, org_near, org_far);
  }
}

///////////////////////////////////////////////////////////////////////////

void xf_set_modelview_matrix(uint32 addr)
{
 int i;
 uint32 crc = 0;
 static uint32 oldcrc = 0;

 // good ole Duddie... checking for changes :)
 for(i = 0 ; i < 12 ; i++)
  {
   crc += gp_xf_regs[addr + i].i;
  }
 if (crc==oldcrc) return;
 oldcrc = crc;

 xf_modelview_matrix[0] = gp_xf_regs[addr + 0].f;
 xf_modelview_matrix[1] = gp_xf_regs[addr + 4].f;
 xf_modelview_matrix[2] = gp_xf_regs[addr + 8].f;
 xf_modelview_matrix[3] = 0.0f;
 xf_modelview_matrix[4] = gp_xf_regs[addr + 1].f;
 xf_modelview_matrix[5] = gp_xf_regs[addr + 5].f;
 xf_modelview_matrix[6] = gp_xf_regs[addr + 9].f;
 xf_modelview_matrix[7] = 0.0f;
 xf_modelview_matrix[8] = gp_xf_regs[addr + 2].f;
 xf_modelview_matrix[9] = gp_xf_regs[addr + 6].f;
 xf_modelview_matrix[10] = gp_xf_regs[addr + 10].f;
 xf_modelview_matrix[11] = 0.0f;
 xf_modelview_matrix[12] = gp_xf_regs[addr + 3].f;
 xf_modelview_matrix[13] = gp_xf_regs[addr + 7].f;
 xf_modelview_matrix[14] = gp_xf_regs[addr + 11].f;
 xf_modelview_matrix[15] = 1.0f;

#ifdef AUX_ILLEGALCALLS
   if(vx_started) auxprintf("illegal MatrixMode (c)!\n");
#endif

 glMatrixMode(GL_MODELVIEW);
 glLoadMatrixf(xf_modelview_matrix);
}

///////////////////////////////////////////////////////////////////////////

void xf_set_normal_matrix(uint32 addr)
{
 // OpenGL does not let setting Normal Matrix
 // it is calculated as Transpose Inverse of Modelview matrix internally in OpenGL
}

///////////////////////////////////////////////////////////////////////////

void xf_set_texture_matrix(uint32 id)
{
 int i;
 uint32 addr;

 addr = xf_texture_matrix_addr[id];

 for(i = 0 ; i < 16 ; i++)
  xf_texture_matrix[id][i] = 0.0f;

 xf_texture_matrix[id][0] = gp_xf_regs[addr + 0].f;
 xf_texture_matrix[id][1] = gp_xf_regs[addr + 4].f;
 xf_texture_matrix[id][2] = gp_xf_regs[addr + 8].f;
 xf_texture_matrix[id][3] = 0.0f;

 xf_texture_matrix[id][4] = gp_xf_regs[addr + 1].f;
 xf_texture_matrix[id][5] = gp_xf_regs[addr + 5].f;
 xf_texture_matrix[id][6] = gp_xf_regs[addr + 9].f;
 xf_texture_matrix[id][7] = 0.0f;

 xf_texture_matrix[id][8] =  gp_xf_regs[addr + 3].f;
 xf_texture_matrix[id][9] =  gp_xf_regs[addr + 7].f;
 xf_texture_matrix[id][10] = gp_xf_regs[addr + 11].f;
 xf_texture_matrix[id][11] = 0.0f;

 xf_texture_matrix[id][12] = gp_xf_regs[addr + 2].f;
 xf_texture_matrix[id][13] = gp_xf_regs[addr + 6].f;
 xf_texture_matrix[id][14] = gp_xf_regs[addr + 10].f;
 xf_texture_matrix[id][15] = 1.0f;

/*
// old:
 xf_texture_matrix[id][0] = gp_xf_regs[addr + 0].f;
 xf_texture_matrix[id][1] = gp_xf_regs[addr + 4].f;
 xf_texture_matrix[id][2] = gp_xf_regs[addr + 8].f;
 xf_texture_matrix[id][3] = 0.0f;

 xf_texture_matrix[id][4] = gp_xf_regs[addr + 1].f;
 xf_texture_matrix[id][5] = gp_xf_regs[addr + 5].f;
 xf_texture_matrix[id][6] = gp_xf_regs[addr + 9].f;
 xf_texture_matrix[id][7] = 0.0f;

 xf_texture_matrix[id][8] = gp_xf_regs[addr + 2].f;
 xf_texture_matrix[id][9] = gp_xf_regs[addr + 6].f;
 xf_texture_matrix[id][10] = gp_xf_regs[addr + 10].f;
 xf_texture_matrix[id][11] = 0.0f;

 xf_texture_matrix[id][12] = gp_xf_regs[addr + 3].f;
 xf_texture_matrix[id][13] = gp_xf_regs[addr + 7].f;
 xf_texture_matrix[id][14] = gp_xf_regs[addr + 11].f;
 xf_texture_matrix[id][15] = 1.0f;
}
*/

#ifdef AUX_ILLEGALCALLS
   if(vx_started) auxprintf("illegal MatrixMode (t)!\n");
#endif

 glActiveTextureARB_Ex(GL_TEXTURE0_ARB + id);

 glMatrixMode(GL_TEXTURE);
 glLoadMatrixf(xf_texture_matrix[id]);
 glMatrixMode(GL_MODELVIEW);

/*
if(GetAsyncKeyState(VK_SHIFT)&32768)
auxprintf("%d; %03d %03d %03d %03d %03d %03d %03d %03d %03d %03d %03d %03d %03d %03d %03d %03d!\n",
        id,
        (int)(xf_texture_matrix[id][0]*1000),
        (int)(xf_texture_matrix[id][1]*1000),
        (int)(xf_texture_matrix[id][2]*1000),
        (int)(xf_texture_matrix[id][3]*1000),
        (int)(xf_texture_matrix[id][4]*1000),
        (int)(xf_texture_matrix[id][5]*1000),
        (int)(xf_texture_matrix[id][6]*1000),
        (int)(xf_texture_matrix[id][7]*1000),
        (int)(xf_texture_matrix[id][8]*1000),
        (int)(xf_texture_matrix[id][9]*1000),
        (int)(xf_texture_matrix[id][10]*1000),
        (int)(xf_texture_matrix[id][11]*1000),
        (int)(xf_texture_matrix[id][12]*1000),
        (int)(xf_texture_matrix[id][13]*1000),
        (int)(xf_texture_matrix[id][14]*1000),
        (int)(xf_texture_matrix[id][15]*1000));
*/
}

///////////////////////////////////////////////////////////////////////////

/*
 1xxx -> L*LDir (diffuse spotlight)
 0xxx -> N*H    (specular)

 x1xx -> use gc atten
 x0xx -> atten =1,0,0

 xx00 -> atten =1,0,0
 xx01 -> N*L      (diffuse)
 xx10 -> N*L[0-1] (diffuse)

 cos0 + cos1 * X + cos2 * X2

*/

#define PI  3.14159265358979323846f

void SetLightAttenuation(int id, uint32 data)
{
 GLfloat a_c,a_l,a_q,mul=1.0f;

 // set light attenuation

 if(data & 0x200)
  {
   GLfloat fc;

   fc=xf_light[id].atten_cos[0].f+(1.0f*xf_light[id].atten_cos[1].f)+(1.0f*xf_light[id].atten_cos[2].f);
   if(fc<=0.0f) fc=0.000001f;
   if(fc>1.0f)  fc=1.0f;
   fc=1.0f/fc;

   fc=1.0f;

   a_c=xf_light[id].atten_dis[0].f * fc * mul;
   a_l=xf_light[id].atten_dis[1].f * fc * mul;
   a_q=xf_light[id].atten_dis[2].f * fc * mul;

   if(a_l<0.0f) {a_c+=a_l;a_l=0.00001f;}
   if(a_q<0.0f) {a_c+=a_q;a_q=0.00001f;}
   if(a_c<0.0f) a_c=0.00001f;

   if((xf_light[id].atten_cos[0].f==1.0f &&
       xf_light[id].atten_cos[1].f==0.0f &&
       xf_light[id].atten_cos[2].f==0.0f) ||
       !(data&0x400))
    {
     glLightf(GL_LIGHT0 + id, GL_SPOT_CUTOFF,   180.0f);
     glLightf(GL_LIGHT0 + id, GL_SPOT_EXPONENT,   0.0f);
     //a_c=2.0f;a_l=0.0f;a_q=0.0f;
    }
   else
    {
     GLfloat fcut,fexp=0.0f;

     // direction

     glLightfv(GL_LIGHT0 + id, GL_SPOT_DIRECTION, (GLfloat *)xf_light[id].dir);

     // pfff... somehow we need to get cutoff/exp from "atten_cos" vals

     if(xf_light[id].atten_cos[1].f==1000.0f)
      {
       //auxprintf("FLAT\n");
       fcut=xf_light[id].atten_cos[0].f/-1000.0f;
      }
     else
     if(xf_light[id].atten_cos[2].f==0.0f &&
        xf_light[id].atten_cos[1].f> 0.0f   &&
        (xf_light[id].atten_cos[0].f+xf_light[id].atten_cos[1].f)>=0.9f)
      {
       //auxprintf("COS\n");
       fcut=1.0f-(1.0f/xf_light[id].atten_cos[1].f);
      }
     else
     if(xf_light[id].atten_cos[0].f==0.0f &&
        xf_light[id].atten_cos[2].f> 0.0f   &&
        (xf_light[id].atten_cos[1].f+xf_light[id].atten_cos[2].f)>=0.9f)
      {
       //auxprintf("COS2\n");
       fcut=1.0f-(1.0f/xf_light[id].atten_cos[2].f);
      }
     else
     if(xf_light[id].atten_cos[1].f> 0.0f &&
        xf_light[id].atten_cos[2].f< 0.0f &&
        (xf_light[id].atten_cos[2].f-xf_light[id].atten_cos[0].f)<=-0.99f &&
        (xf_light[id].atten_cos[2].f-xf_light[id].atten_cos[0].f)>=-1.01f)

      {
       //auxprintf("SHARP\n");
       fcut=1.0f-sqrt((-1.0f/xf_light[id].atten_cos[2].f));
      }
     else
     if(xf_light[id].atten_cos[1].f> 0.0f &&
        xf_light[id].atten_cos[0].f<= 0.0f &&
        xf_light[id].atten_cos[2].f<xf_light[id].atten_cos[0].f)
      {
       //auxprintf("RING1+2\n");
       fcut=1.0f-sqrt((-4.0f/xf_light[id].atten_cos[2].f));
      }
     else                         
      {
       //bla
       fcut=PI/2.0f;
       fexp=20.0f;
      }

     if(fcut<-1.0f) fcut=-1.0f;
     if(fcut> 1.0f) fcut= 1.0f;
     fcut=(acos(fcut)*180.0f)/PI;

     //auxprintf("-> %d\n",(int)(fcut*1000.0f));

     glLightf(GL_LIGHT0 + id, GL_SPOT_CUTOFF,  fcut);
     glLightf(GL_LIGHT0 + id, GL_SPOT_EXPONENT,fexp);

/*
     glLightfv(GL_LIGHT0 + id, GL_SPOT_DIRECTION, (GLfloat *)xf_light[id].dir);

     // pfff... somehow we need to get cutoff/exp from "atten_cos" vals

     glLightf(GL_LIGHT0 + id, GL_SPOT_CUTOFF, 90.0f);
     glLightf(GL_LIGHT0 + id, GL_SPOT_EXPONENT,   20.0f);
     //glLightf(GL_LIGHT0 + id, GL_SPOT_EXPONENT,   2.0f);
     //a_c=1.0f;a_l=0.0f;a_q=0.0f;
*/
    }
  }
 else
  {  
   a_c=1.0f*mul;a_l=0.0f;a_q=0.0f;

   glLightf(GL_LIGHT0 + id, GL_SPOT_CUTOFF,   180.0f);
   glLightf(GL_LIGHT0 + id, GL_SPOT_EXPONENT,   0.0f);
  }

//auxprintf("%d -> %d,%d,%d\n",id,(int)(a_c*1000.0f),(int)(a_l*1000.0f),(int)(a_q*1000.0f));

 if(curr_light_attn[id][0]!=a_c ||
    curr_light_attn[id][1]!=a_l ||
    curr_light_attn[id][2]!=a_q)
  {                                    
   curr_light_attn[id][0]=a_c;                         
   curr_light_attn[id][1]=a_l;
   curr_light_attn[id][2]=a_q;

//auxprintf("%d(%08X) -> %d,%d,%d\n",id,data,(int)(a_c*1000.0f),(int)(a_l*1000.0f),(int)(a_q*1000.0f));
//auxprintf("cos: %d,%d,%d\n",(int)(xf_light[id].atten_cos[0].f*1000.0f),(int)(xf_light[id].atten_cos[1].f*1000.0f),(int)(xf_light[id].atten_cos[2].f*1000.0f));
//auxprintf("dis: %d,%d,%d\n",(int)(xf_light[id].atten_dis[0].f*1000.0f),(int)(xf_light[id].atten_dis[1].f*1000.0f),(int)(xf_light[id].atten_dis[2].f*1000.0f));

   glLightf(GL_LIGHT0 + id,GL_CONSTANT_ATTENUATION, a_c);
   glLightf(GL_LIGHT0 + id,GL_LINEAR_ATTENUATION,   a_l);
   glLightf(GL_LIGHT0 + id,GL_QUADRATIC_ATTENUATION,a_q);
  }
}

///////////////////////////////////////////////////////////////////////////

void xf_update_chanctrl(uint32 chan)
{
 uint32 data = gp_xf_regs[0x100e].i;

 // PETE: way too much unneccessary calls... we will change that, I think
                                                      
//if(GetAsyncKeyState(VK_SHIFT)&32768)
// auxprintf("-> %08X\n",data);
// 706,736,506


 //auxprintf("-> %08X\n",(data & 0x1) | ((data >> 5) & 0x2));

 switch((data & 0x1) | ((data >> 5) & 0x2))
  {
   case 0x3:   // AMB == 1 && DIF == 1
    //syslog(XF,"COL0CTRL Enable Material Color for Ambient and Diffuse\n");
/*
    glColorMaterial(GL_FRONT, GL_AMBIENT_AND_DIFFUSE);
    glEnable(GL_COLOR_MATERIAL);
*/

    glColorMaterial(GL_FRONT, GL_AMBIENT_AND_DIFFUSE);
    glEnable(GL_COLOR_MATERIAL);
    break;

   case 0x2:   // AMB == 1 && DIF == 0
    //syslog(XF,"COL0CTRL Enable Material Color for Ambient\n");
    // diffuse comes from COL0MAT register XF_100c
/*
    glMaterialfv(GL_FRONT, GL_DIFFUSE, mat_col);
    glColorMaterial(GL_FRONT, GL_AMBIENT);
    glEnable(GL_COLOR_MATERIAL);
    glColor4fv(col0_amb);
*/

    glMaterialfv(GL_FRONT, GL_DIFFUSE, mat_col);
    glColorMaterial(GL_FRONT, GL_AMBIENT);
    glEnable(GL_COLOR_MATERIAL);
    glColor4fv(col0_amb);
    break;

   case 0x1:   // AMB == 0 && DIF == 1
    //syslog(XF,"COL0CTRL Enable Material Color for Diffuse\n");
    // ambient comes from COL0AMB register XF_100a
/*
    glMaterialfv(GL_FRONT, GL_AMBIENT, col0_amb);
    glColorMaterial(GL_FRONT, GL_DIFFUSE);
    glEnable(GL_COLOR_MATERIAL);
    glColor4fv(mat_col);
*/
    glMaterialfv(GL_FRONT, GL_AMBIENT, col0_amb);
    glColorMaterial(GL_FRONT, GL_DIFFUSE);
    glEnable(GL_COLOR_MATERIAL);
    glColor4fv(mat_col);

    break;

   case 0x0:   // AMB == 0 && DIF == 0
    //syslog(XF,"COL0CTRL Disable Material Color\n");
    // both colours come from their registers
/*
    glEnable(GL_COLOR_MATERIAL);
    glColorMaterial(GL_FRONT_AND_BACK, GL_AMBIENT);
    glColor4fv(col0_amb);
    glColorMaterial(GL_FRONT_AND_BACK, GL_DIFFUSE);
    glColor4fv(mat_col);
    glColorMaterial(GL_FRONT_AND_BACK, GL_AMBIENT_AND_DIFFUSE);
    glEnable(GL_COLOR_MATERIAL);

auxprintf("amb %d,%d,%d,%d\n",(int)(col0_amb[0]*1000.0f),(int)(col0_amb[1]*1000.0f),(int)(col0_amb[2]*1000.0f),(int)(col0_amb[3]*1000.0f));
auxprintf("mat %d,%d,%d,%d\n",(int)(mat_col[0]*1000.0f),(int)(mat_col[1]*1000.0f),(int)(mat_col[2]*1000.0f),(int)(mat_col[3]*1000.0f));
*/

    glMaterialfv(GL_FRONT, GL_AMBIENT, col0_amb);
    glMaterialfv(GL_FRONT, GL_DIFFUSE, mat_col);
    glDisable(GL_COLOR_MATERIAL);
    glColor4fv(mat_col);

    break;
  }                                                   

 //----------------------------------------------------//
 if (data & 0x2)
  {
   //syslog(XF,"COL0CTRL Enable Lighting\n");
   glEnable(GL_LIGHTING);
   glEnable(GL_NORMALIZE);
  }
 else
  {
   //syslog(XF,"COL0CTRL Disable Lighting\n");
   glDisable(GL_LIGHTING);
  }
 //----------------------------------------------------//
 if (data & (1 << 2))
  {
   SetLightAttenuation(0,data);
   //syslog(XF,"COL0CTRL Enable Light 0\n");
   glEnable(GL_LIGHT0);
  }
 else
  {
   //syslog(XF,"COL0CTRL Disable Light 0\n");
   glDisable(GL_LIGHT0);
  }
 //----------------------------------------------------//
 if (data & (1 << 3))
  {
   SetLightAttenuation(1,data);
   //syslog(XF,"COL0CTRL Enable Light 1\n");
   glEnable(GL_LIGHT1);
  }
 else
  {
   //syslog(XF,"COL0CTRL Disable Light 1\n");
   glDisable(GL_LIGHT1);
  }
 //----------------------------------------------------//
 if (data & (1 << 4))
  {
   SetLightAttenuation(2,data);
   //syslog(XF,"COL0CTRL Enable Light 2\n");
   glEnable(GL_LIGHT2);
  }
 else
  {
   //syslog(XF,"COL0CTRL Disable Light 2\n");
   glDisable(GL_LIGHT2);
  }
 //----------------------------------------------------//
 if (data & (1 << 5))
  {
   SetLightAttenuation(3,data);
   //syslog(XF,"COL0CTRL Enable Light 3\n");
   glEnable(GL_LIGHT3);
  }
 else
  {
   //syslog(XF,"COL0CTRL Disable Light 3\n");
   glDisable(GL_LIGHT3);
  }
 //----------------------------------------------------//
 if (data & (1 << 11))
  {
   SetLightAttenuation(4,data);
   //syslog(XF,"COL0CTRL Enable Light 4\n");
   glEnable(GL_LIGHT4);
  }
 else
  {
   //syslog(XF,"COL0CTRL Disable Light 4\n");
   glDisable(GL_LIGHT4);
  }
 //----------------------------------------------------//
 if (data & (1 << 12))
  {
   SetLightAttenuation(5,data);
   //syslog(XF,"COL0CTRL Enable Light 5\n");
   glEnable(GL_LIGHT5);
  }
 else
  {
   //syslog(XF,"COL0CTRL Disable Light 5\n");
   glDisable(GL_LIGHT5);
  }
 //----------------------------------------------------//
 if (data & (1 << 13))
  {
   SetLightAttenuation(6,data);
   //syslog(XF,"COL0CTRL Enable Light 6\n");
   glEnable(GL_LIGHT6);
  }
 else
  {
   //syslog(XF,"COL0CTRL Disable Light 6\n");
   glDisable(GL_LIGHT6);
  }
 //----------------------------------------------------//
 if (data & (1 << 14))
  {
   SetLightAttenuation(7,data);
   //syslog(XF,"COL0CTRL Enable Light 7\n");
   glEnable(GL_LIGHT7);
  }
 else
  {
   //syslog(XF,"COL0CTRL Disable Light 7\n");
   glDisable(GL_LIGHT7);
  }
 //----------------------------------------------------//
}

///////////////////////////////////////////////////////////////////////////

void gp_xf_write_reg(uint32 reg, uint32 data)
{
 gp_xf_regs[reg].i = data;

 //syslog(XF,"GP: XF_%04x %08x %f\n", reg, data, gp_xf_regs[reg].f);

 // for now we want switch statement but later we need special callbacks next to registers

 if (reg >= xf_texture_matrix_addr[0] && reg <= (xf_texture_matrix_addr[0] + 8))
  {
   xf_set_texture_matrix(0);
  }

 if (reg >= xf_texture_matrix_addr[1] && reg <= (xf_texture_matrix_addr[1] + 8))
  {
   xf_set_texture_matrix(1);
  }

 if (reg >= xf_texture_matrix_addr[2] && reg <= (xf_texture_matrix_addr[2] + 8))
  {
   xf_set_texture_matrix(2);
  }

 if (reg >= xf_texture_matrix_addr[3] && reg <= (xf_texture_matrix_addr[3] + 8))
  {
   xf_set_texture_matrix(3);
  }

 if (reg >= xf_texture_matrix_addr[4] && reg <= (xf_texture_matrix_addr[4] + 8))
  {
   xf_set_texture_matrix(4);
  }

 if (reg >= xf_texture_matrix_addr[5] && reg <= (xf_texture_matrix_addr[5] + 8))
  {
   xf_set_texture_matrix(5);
  }

 if (reg >= xf_texture_matrix_addr[6] && reg <= (xf_texture_matrix_addr[6] + 8))
  {
   xf_set_texture_matrix(6);
  }

 if (reg >= xf_texture_matrix_addr[7] && reg <= (xf_texture_matrix_addr[7] + 8))
  {
   xf_set_texture_matrix(7);
  }

 if (reg >= xf_modelview_matrix_addr && reg <= (xf_modelview_matrix_addr + 11))
  {
   xf_set_modelview_matrix(xf_modelview_matrix_addr);
  }

 if (reg >= xf_normal_matrix_addr && reg <= (xf_normal_matrix_addr + 11))
  {
   xf_set_normal_matrix(xf_normal_matrix_addr);
  }

 if(reg >= 0x600 && reg < 0x680)
  {
   // light number
   uint32 id = (reg >> 4) & 0xf;

   reg &= 0xf;
   switch(reg)
    {
     case 0x3:
      // $LIGHT$ COLOUR
// PETE: 255 instead 256
/*
      xf_light[id].col[0] = ((data >> 24) & 0xff)/512.0f;//255.0f;
      xf_light[id].col[1] = ((data >> 16) & 0xff)/512.0f;//255.0f;
      xf_light[id].col[2] = ((data >> 8) & 0xff)/512.0f;//255.0f;
      xf_light[id].col[3] = ((data >> 0) & 0xff)/512.0f;//255.0f;
*/

      xf_light[id].col[0] = ((data >> 24) & 0xff)/255.0f;
      xf_light[id].col[1] = ((data >> 16) & 0xff)/255.0f;
      xf_light[id].col[2] = ((data >> 8) & 0xff)/255.0f;
      xf_light[id].col[3] = ((data >> 0) & 0xff)/255.0f;

      if(xf_light[id].col[0] != curr_light_color[id][0] ||
         xf_light[id].col[1] != curr_light_color[id][1] ||
         xf_light[id].col[2] != curr_light_color[id][2] ||
         xf_light[id].col[3] != curr_light_color[id][3])
       {
    //    GLfloat gv[4]={0.0f,0.0f,0.0f,0.0f};
    //    glLightfv(GL_LIGHT0 + id, GL_AMBIENT, gv);

        curr_light_color[id][0]=xf_light[id].col[0];
        curr_light_color[id][1]=xf_light[id].col[1];
        curr_light_color[id][2]=xf_light[id].col[2];
        curr_light_color[id][3]=xf_light[id].col[3];
        
        glLightfv(GL_LIGHT0 + id, GL_DIFFUSE, xf_light[id].col);
/*
      xf_light[id].col[0]/=5.0f;
      xf_light[id].col[1]/=5.0f;
      xf_light[id].col[2]/=5.0f;
        glLightfv(GL_LIGHT0 + id, GL_AMBIENT, xf_light[id].col);
*/

 //       glLightfv(GL_LIGHT0 + id, GL_SPECULAR, xf_light[id].col);
       }
                         
//      glLightfv(GL_LIGHT0 + id, GL_DIFFUSE, xf_light[id].col);
//      glLightfv(GL_LIGHT0 + id, GL_AMBIENT, xf_light[id].col);
//      glLightfv(GL_LIGHT0 + id, GL_SPECULAR, xf_light[id].col);
      break;

     case 0x4:
     case 0x5:
     case 0x6:
      xf_light[id].atten_cos[reg - 0x4].i = data;
//    auxprintf("a %d,%d cos %d\n",id,reg - 0x4,
//              (int)(xf_light[id].atten_cos[reg - 0x4].f*1000.0f));
      break;

     case 0x7:
     case 0x8:
     case 0x9:
      xf_light[id].atten_dis[reg - 0x7].i = data;
//    auxprintf("k %d,%d dis %d\n",id,reg - 0x7,
//                (int)(xf_light[id].atten_dis[reg - 0x7].f*1000.0f));
      break;

     case 0xa:
     case 0xb:
     case 0xc:
      // $LIGHT$ POSITION
      xf_light[id].pos[reg - 0xa].i = data;
      xf_light[id].pos[3].f = 1.0f;

#ifdef AUX_ILLEGALCALLS
   if(vx_started) auxprintf("illegal MatrixMode (d)!\n");
#endif

      // opengl: light pos is relative to modelview 
      glMatrixMode(GL_MODELVIEW);
      glPushMatrix();
      glLoadIdentity();
      glLightfv(GL_LIGHT0 + id, GL_POSITION, (GLfloat *)xf_light[id].pos);
      glPopMatrix();

      break;

     case 0xd:
     case 0xe:
     case 0xf:
      // $LIGHT$ DIRECTION
      xf_light[id].dir[reg - 0xd].i = data;
      xf_light[id].dir[3].f = 1.0f;
      // !test!
      xf_light[id].dir[reg - 0xd].f=-xf_light[id].dir[reg - 0xd].f;

#ifdef AUX_ILLEGALCALLS
   if(vx_started) auxprintf("illegal MatrixMode (e)!\n");
#endif
/*
      glMatrixMode(GL_MODELVIEW);
      glPushMatrix();
      glLoadIdentity();
      //glLightfv(GL_LIGHT0 + id, GL_SPOT_DIRECTION, (GLfloat *)xf_light[id].dir);
      //glLightf(GL_LIGHT0 + id, GL_SPOT_CUTOFF, 100.0f);
      glPopMatrix();
*/
     break;

     default:
      //syslog(XF,"LIGHT: Unknown register\n");
      break;
    }
   return;
  }


 switch(reg)                                          
  {//--------------------------------------------------//
   case 0x100a:
    // ambient colour 0

    col0_amb[0] = ((data >> 24) & 0xff)/255.0f;
    col0_amb[1] = ((data >> 16) & 0xff)/255.0f;
    col0_amb[2] = ((data >> 8) & 0xff)/255.0f;
    col0_amb[3] = ((data >> 0) & 0xff)/255.0f;

    xf_update_chanctrl(0);
    break;
   //--------------------------------------------------//
  case 0x100b:
   // ambient colour 1
/*
auxprintf("amb1 %d,%d,%d,%d\n",
 ((data >> 24) & 0xff),
 ((data >> 16) & 0xff),
 ((data >> 8) & 0xff),
 ((data >> 0) & 0xff));
*/
    break;
   //--------------------------------------------------//
   case 0x100c:
    // material colour 0
    mat_col[0] = ((data >> 24) & 0xff)/255.0f;
    mat_col[1] = ((data >> 16) & 0xff)/255.0f;
    mat_col[2] = ((data >> 8) & 0xff)/255.0f;
    mat_col[3] = ((data >> 0) & 0xff)/255.0f;
    xf_update_chanctrl(0);
    break;
   //--------------------------------------------------//
 case 0x100d:
    // material colour 1
/*
auxprintf("mat1 %d,%d,%d,%d\n",
 ((data >> 24) & 0xff),
 ((data >> 16) & 0xff),
 ((data >> 8) & 0xff),
 ((data >> 0) & 0xff));
*/
    break;
   //--------------------------------------------------//
   case 0x100e:
    // colour 0 control
    xf_update_chanctrl(0);
    break;
   //--------------------------------------------------//
// case 0x100f: colour 1 control
// case 0x1010: alpha 0 cntrol
// case 0x1011: alpha 1 cntrol
   //--------------------------------------------------//
   case 0x1018:
    //syslog(XF,"SetMatrix: MV %x T0: %x T1: %x T2: %x T3: %x\n", data & 0x3f, (data >> 6) & 0x3f, (data >> 12) & 0x3f, (data >> 18) & 0x3f, (data >> 24) & 0x3f);

    xf_modelview_matrix_addr = (data & 0x3f) << 2;
    xf_set_modelview_matrix(xf_modelview_matrix_addr);

    xf_texture_matrix_addr[0] = (((data >> 6) & 0x3f) << 2);
    xf_texture_matrix_addr[1] = (((data >> 12) & 0x3f) << 2);
    xf_texture_matrix_addr[2] = (((data >> 18) & 0x3f) << 2);
    xf_texture_matrix_addr[3] = (((data >> 24) & 0x3f) << 2);
    xf_set_texture_matrix(0);
    xf_set_texture_matrix(1);
    xf_set_texture_matrix(2);
    xf_set_texture_matrix(3);
    break;
   //--------------------------------------------------//
   case 0x1019:
    xf_texture_matrix_addr[4] = (((data >> 0) & 0x3f) << 2);
    xf_texture_matrix_addr[5] = (((data >> 6) & 0x3f) << 2);
    xf_texture_matrix_addr[6] = (((data >> 12) & 0x3f) << 2);
    xf_texture_matrix_addr[7] = (((data >> 18) & 0x3f) << 2);
    xf_set_texture_matrix(4);
    xf_set_texture_matrix(5);
    xf_set_texture_matrix(6);
    xf_set_texture_matrix(7);
    break;
   //--------------------------------------------------//
   case 0x1026:
    // write to PROJ_MTX_MODE
    xf_set_projection_matrix(data);
    break;
   //--------------------------------------------------//
/*

// it seems: we hope, that a 0x101f will always be at the end

case 0x101a:
case 0x101b:
case 0x101c:
case 0x101d:
case 0x101e:
*/


   case 0x101f:
    {
     float x, y, w, h;
     float zfar, znear;

     w = gp_xf_regs[0x101a].f;
     h = -gp_xf_regs[0x101b].f;

     x = gp_xf_regs[0x101d].f;
     y = gp_xf_regs[0x101e].f;

     x = x - 342 - w;
     y = y - 342 - h;
     w *= 2.0f;
     h = h * 2.0f;

//      y = 480 - y;
//      h = -h;

     y = 480 - (y + h);

//      printf("org: %f, %f, %f, %f\n", gp_xf_regs[0x101a].f, gp_xf_regs[0x101b].f, gp_xf_regs[0x101d].f, gp_xf_regs[0x101e].f);
//      syslog(XF,"glViewport(%d, %d, %d, %d)\n", (GLint)x, (GLint)y, (GLsizei)w, (GLsizei)h);

     glViewport((GLint)x, (GLint)y, (GLsizei)w, (GLsizei)h);
        
     zfar = gp_xf_regs[0x101f].f;
     znear = gp_xf_regs[0x101c].f - zfar;
     zfar /= 16777215.0f;
     znear /= 16777215.0f;

 //      syslog(XF,"Depth Range %f %f\n", znear, zfar);

     glDepthRange(znear, zfar);
     
    } break;
   //--------------------------------------------------//
   case 0x1008:
    if (data & 0xf0)
     {
//          glEnable(GL_TEXTURE_2D);
     }
    else
     {
//          glDisable(GL_TEXTURE_2D);
     }
    break;
   //--------------------------------------------------//
   case 0x103f:
    // texgen enabled
    // bit 0 enables texgen matrix

/*      if (data)
        {
            glTexGeni(GL_S, GL_TEXTURE_GEN_MODE, GL_SPHERE_MAP);
            glTexGeni(GL_T, GL_TEXTURE_GEN_MODE, GL_SPHERE_MAP);
            glEnable(GL_TEXTURE_GEN_S);
            glEnable(GL_TEXTURE_GEN_T);
        }
        else
        {
            glDisable(GL_TEXTURE_GEN_S);
            glDisable(GL_TEXTURE_GEN_T);
        }
*/ 
    break;
   //--------------------------------------------------//
   default:
    break;
   //--------------------------------------------------//
  }
}

///////////////////////////////////////////////////////////////////////////

