/***************************************************************************
                    fragment_shader_gen.c  -  description
                             -------------------
    begin                : 
    copyright            : (C) 2005 by Duddie
    email                : duddie@walla.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version. See also the license.txt file for *
 *   additional informations.                                              *
 *                                                                         *
 ***************************************************************************/

//*************************************************************************//
// History of changes:
//
// 2005/06/06 - Pete
// - added cache_clear function (to prevent crashes)
//
// 2005/06/05 - Pete
// - added fog to shader generator
//
//*************************************************************************//

#include "stdafx.h"
#include <stdio.h>
#include "texture_environment.h"
#include "bp_regs.h"
#include "fragment_shader_gen.h"
#define _IN_FSGEN
#include "externals.h"

///////////////////////////////////////////////////////////////////////////
// GLOBALS + DEFINES

static char     fsg_program_source[10*1024];
static GLuint   fsg_fragment_program;
static uint32   fsg_icount;
static uint32   fsg_icount_peak;

#ifdef AUX_SHADER
#define FS_ADD(X)       fsg_icount++;strcat(fs, X);strcat(fs,"\n");auxprintf(X);auxprintf("\n");
#define FS_ADD_VAR(X)   strcat(fs, X);strcat(fs,"\n");auxprintf(X);auxprintf("\n");
#define FS_ADD_COMMENT(X)   strcat(fs, "## ");strcat(fs, X);strcat(fs, "\n");auxprintf("## ");auxprintf(X);auxprintf("\n");
#else
#define FS_ADD(X)       fsg_icount++;strcat(fs, X);strcat(fs,"\n");
#define FS_ADD_VAR(X)   strcat(fs, X);strcat(fs,"\n");
#define FS_ADD_COMMENT(X)   strcat(fs, "## ");strcat(fs, X);strcat(fs, "\n");
#endif

#define FS_INIT()   fs[0] = 0
#define FS_HEADER() FS_ADD("!!ARBfp1.0")
#define FS_END()    FS_ADD("END");

enum
{
    CHAN_A = 0,
    CHAN_B,
    CHAN_C,
    CHAN_D
};


uint32  gstage;

uint32  cc_used;
uint32  ca_used;

#define TE_MAX_STAGES               16
#define FSG_CACHE_SLICE_ENTRIES     32

typedef struct
{
    uint32  *combiner_regs;
    uint16  *stage;
    uint16  *konst;
    uint32  paletted;
    uint8   fog;
    GLuint  fragment_program;
} fsg_cache_entry_t;

fsg_cache_entry_t   **fsg_cache[TE_MAX_STAGES];

uint32  fsg_cache_entries[TE_MAX_STAGES];
uint32  fsg_cache_slots[TE_MAX_STAGES];

#define FSG_DUAL_CH(X, Y)   ((X << 4) | Y)

///////////////////////////////////////////////////////////////////////////
// CODE

void fsg_cache_init(void)
{
 uint32 i;

 for(i = 0 ; i < TE_MAX_STAGES ; i++)
  {
   fsg_cache[i] = (fsg_cache_entry_t **) malloc(sizeof(fsg_cache_entry_t *) * FSG_CACHE_SLICE_ENTRIES);
   fsg_cache_slots[i] = FSG_CACHE_SLICE_ENTRIES;
   fsg_cache_entries[i] = 0;
  }
}

///////////////////////////////////////////////////////////////////////////

void fsg_cache_clear(uint8 stages)
{
 fsg_cache_entry_t   *current;
 uint32 j;

 glDisable(GL_FRAGMENT_PROGRAM_ARB);                   // to be sure: deactivate current shader
 glBindProgramARB_Ex(GL_FRAGMENT_PROGRAM_ARB,0);
 
 for(j = 0 ; j < fsg_cache_entries[stages] ; j++)
  {
   current = fsg_cache[stages][j];
 
   free(current->combiner_regs);
   free(current->stage);
   free(current->konst);

   if(current->fragment_program)
    glDeleteProgramsARB_Ex(1,&current->fragment_program);
   free(current);

   fsg_cache[stages][j]=0;
  }

 fsg_cache_entries[stages]=0;
}

///////////////////////////////////////////////////////////////////////////

void fsg_cache_add(GLuint fragment_program)
{
 uint8   stages;
 uint16  entry;
 uint32  i;
 fsg_cache_entry_t   *current;

 stages = te_active_stages;
    
 entry = fsg_cache_entries[stages];

 if (entry >= fsg_cache_slots[stages])
  {
   // entries have overgrown slots
   //gpu_exit(0,0);

   fsg_cache_clear(stages);
   entry=0;              
  }

 fsg_cache_entries[stages]++;
 fsg_cache[stages][entry] = (fsg_cache_entry_t *)malloc(sizeof(fsg_cache_entry_t));

 current = fsg_cache[stages][entry];
 current->combiner_regs = (uint32 *) malloc(TE_COMBINER_REGS_SIZE * stages * 2); // *2 because alpha and color stages
 current->stage = (uint16 *) malloc(sizeof(uint16) * stages);
 current->konst = (uint16 *) malloc(sizeof(uint16) * stages);

 for(i = 0 ; i < stages ; i++)
  {
   current->combiner_regs[i * 2 + 0] = te_combiner_regs[i * 2 + 0];
   current->combiner_regs[i * 2 + 1] = te_combiner_regs[i * 2 + 1];
   current->stage[i] = tev_stage[i];
   current->konst[i] = tev_konst[i];
  }

 current->paletted = tx_paletted;
 current->fog      = bp_fog;
 current->fragment_program = fragment_program;

 // fprintf(stderr, "Added entry: %d @ %d\n", entry, stages);
}

///////////////////////////////////////////////////////////////////////////

void fsg_cache_print(void)
{
}

///////////////////////////////////////////////////////////////////////////

bool fsg_cache_check(GLuint *fragment_program)
{
 fsg_cache_entry_t   *current;
 uint8   stages;
 uint32  i, j;
 bool    found;

 stages = te_active_stages;
 found = false;

 for(j = 0 ; j < fsg_cache_entries[stages] ; j++)
  {
   current = fsg_cache[stages][j];
   found = true;

   for(i = 0 ; i < stages ; i++)
    {
     if (current->combiner_regs[i * 2 + 0] != te_combiner_regs[i * 2 + 0])
      {
       found = false;
       break;
      }

     if (current->combiner_regs[i * 2 + 1] != te_combiner_regs[i * 2 + 1])
      {
       found = false;
       break;
      }

     if (current->stage[i] != tev_stage[i])
      {
       found = false;
       break;
      }
 
     if (current->konst[i] != tev_konst[i])
      {
       found = false;
       break;
      }

     if (current->paletted != tx_paletted)
      {
       found = false;
       break;
      }

     if (current->fog != bp_fog)
      {
       found = false;
       break;
      }
    }

   if(found == true)
    {
     *fragment_program = current->fragment_program;
     return true;
    }
  }

 return found;
}

///////////////////////////////////////////////////////////////////////////

void fsg_init(void)
{
 fsg_cache_init();
 fsg_icount_peak = 0;
}

///////////////////////////////////////////////////////////////////////////

static void fsg_generate_alpha_chan(char *fs, uint8 input, uint8 chan)
{
 char tmpstr[256];
 uint32  texture_id;

 switch(input)                                         
  {//--------------------------------------------------//
   case TE_CA_APREV:
    if((chan == CHAN_D) && !(ca_used & 0x1))
     {
      FS_ADD("MOV R0.w, env_prev_s10.w;");
     }
    else
     {
      FS_ADD("MOV R0.w, cc_prev.w;");
     }
    break;
   //--------------------------------------------------//
   case TE_CA_A0:
    if((chan == CHAN_D) && !(ca_used & 0x2))
     {
      FS_ADD("MOV R0.w, env_c0_s10.w;");
     }
    else
     {
      FS_ADD("MOV R0.w, cc_c0.w;");
     }
    break;
   //--------------------------------------------------//
   case TE_CA_A1:
    if((chan == CHAN_D) && !(ca_used & 0x4))
     {
      FS_ADD("MOV R0.w, env_c1_s10.w;");
     }
    else
     {
      FS_ADD("MOV R0.w, cc_c1.w;");
     }
    break;
   //--------------------------------------------------//
   case TE_CA_A2:
    if((chan == CHAN_D) && !(ca_used & 0x8))
     {
      FS_ADD("MOV R0.w, env_c2_s10.w;");
     }
    else
     {
      FS_ADD("MOV R0.w, cc_c2.w;");
     }
    break;
   //--------------------------------------------------//
   case TE_CA_TEXA:

    texture_id = (tev_stage[gstage] >> 0) & 0x7;
    
    if (tx_paletted & (1 << texture_id))
     {
      sprintf(tmpstr, "TEX R3, fragment.texcoord[%d], texture[%d], 2D;", (tev_stage[gstage] >> 3) & 0x7, (tev_stage[gstage] >> 0) & 0x7);
      FS_ADD(tmpstr);
      sprintf(tmpstr, "TEX R0.w, R3, texture[%d], 1D;", texture_id + 8);
      FS_ADD(tmpstr);
     }
    else
     {
      sprintf(tmpstr, "TEX R0.w, fragment.texcoord[%d], texture[%d], 2D;", (tev_stage[gstage] >> 3) & 0x7, (tev_stage[gstage] >> 0) & 0x7);
      FS_ADD(tmpstr);
     }
    break;
   //--------------------------------------------------//
   case TE_CA_RASA:
    FS_ADD("MOV R0.w, fragment.color.primary.w;");
    break;
   //--------------------------------------------------//
   case TE_CA_KONST:
    switch((tev_konst[gstage] >> 5) & 0x1f)
     {//-----------------------------------------------//
      case TE_KASEL_K0_A:
       if(chan == CHAN_D)
        {
         FS_ADD("MOV R0.w, env_k0_s10.w;");
        }
       else
        {
         FS_ADD("MOV R0.w, env_k0.w;");
        }
       break;
      //-----------------------------------------------//
      case TE_KASEL_K1_A:
       if(chan == CHAN_D)
        {
         FS_ADD("MOV R0.w, env_k1_s10.w;");
        }
       else
        {
         FS_ADD("MOV R0.w, env_k1.w;");
        }
       break;
      //-----------------------------------------------//
      case TE_KASEL_K2_A:
       if(chan == CHAN_D)
        {
         FS_ADD("MOV R0.w, env_k2_s10.w;");
        }
       else
        {
         FS_ADD("MOV R0.w, env_k2.w;");
        }
       break;
      //-----------------------------------------------//
      case TE_KASEL_K3_A:
       if(chan == CHAN_D)
        {
         FS_ADD("MOV R0.w, env_k3_s10.w;");
        }
       else
        {
         FS_ADD("MOV R0.w, env_k3.w;");
        }
       break;
      //-----------------------------------------------//
      case TE_KASEL_1_2:
       FS_ADD("MOV R0.w, half.w;");
       break;
      //-----------------------------------------------//
      case TE_KASEL_1:
       FS_ADD("MOV R0.w, one.w;");
       break;
      //-----------------------------------------------//
      default:
        //fprintf(stderr, "KONST %s", te_kasel[(tev_konst[gstage] >> 5) & 0x1f]);
        //  gpu_exit(0,1);
        break;
      //-----------------------------------------------//
     }
    break;
   //--------------------------------------------------//
   case TE_CA_ZERO:
    FS_ADD("MOV R0.w, zero.w;");
    break;
   //--------------------------------------------------//
   default:
    FS_ADD("MOV R0.w, zero.w;");

    // fprintf(stderr, "A: Unsupported input %d\n", input);
    // gpu_exit(0,2);
    break;
   //--------------------------------------------------//
  }
}

///////////////////////////////////////////////////////////////////////////

static void fsg_generate_color_chan(char *fs, uint8 input, uint8 chan)
{
 char tmpstr[256];
 uint32  texture_id;

 switch(input)
  {//--------------------------------------------------//
   case TE_CC_TEXC:
    sprintf(tmpstr, "TEX R0, fragment.texcoord[%d], texture[%d], 2D;",(tev_stage[gstage] >> 3) & 0x7, (tev_stage[gstage] >> 0) & 0x7);
    FS_ADD(tmpstr);
//#if WITH_PALETTED_AT_SHADER
    texture_id = (tev_stage[gstage] >> 0) & 0x7;

//auxprintf("-------------TID %d,%d,%d\n",texture_id,tev_stage[gstage],gstage);

    if (tx_paletted & (1 << texture_id))
     {
      sprintf(tmpstr, "TEX R0, R0, texture[%d], 1D;", texture_id + 8);
      FS_ADD(tmpstr);
     }
//#endif
    break;
   //--------------------------------------------------//
   case TE_CC_TEXA:
    // pay attention, because later TEXC must be related to stage
    sprintf(tmpstr, "TEX R0, fragment.texcoord[%d], texture[%d], 2D;", (tev_stage[gstage] >> 3) & 0x7, (tev_stage[gstage] >> 0) & 0x7);
    FS_ADD(tmpstr);
//#if WITH_PALETTED_AT_SHADER
    texture_id = (tev_stage[gstage] >> 0) & 0x7;
    if (tx_paletted & (1 << texture_id))
     {
      sprintf(tmpstr, "TEX R0, R0, texture[%d], 1D;", texture_id + 8);
      FS_ADD(tmpstr);
     }
//#endif
    FS_ADD("MOV R0, R0.wwww;");
    break;
   //--------------------------------------------------//
   case TE_CC_CPREV:
    if((chan == CHAN_D) && !(cc_used & 0x1))
     {
      FS_ADD("MOV R0, env_prev_s10;");
     }
    else
     {
      FS_ADD("MOV R0, cc_prev;");
     }
    break;
   //--------------------------------------------------//
   case TE_CC_APREV:
    if((chan == CHAN_D) && !(cc_used & 0x1))
     {
      FS_ADD("MOV R0, env_prev_s10.wwww;");
     }
    else
     {
      FS_ADD("MOV R0, cc_prev.wwww;");
     }
    break;
   //--------------------------------------------------//
   case TE_CC_C0:
    if((chan == CHAN_D) && !(cc_used & 0x2))
     {
      FS_ADD("MOV R0, env_c0_s10;");
     }
    else
     {
      FS_ADD("MOV R0, cc_c0;");
     }
    break;
   //--------------------------------------------------//
   case TE_CC_A0:
    if((chan == CHAN_D) && !(cc_used & 0x2))
     {
      FS_ADD("MOV R0, env_c0_s10.wwww;");
     }
    else
     {
      FS_ADD("MOV R0, cc_c0.wwww;");
     }
    break;
   //--------------------------------------------------//
   case TE_CC_C1:
    if((chan == CHAN_D) && !(cc_used & 0x4))
     {
      FS_ADD("MOV R0, env_c1_s10;");
     }
    else
     {
      FS_ADD("MOV R0, cc_c1;");
     }
    break;
   //--------------------------------------------------//
   case TE_CC_C2:
    if((chan == CHAN_D) && !(cc_used & 0x8))
     {
      FS_ADD("MOV R0, env_c2_s10;");
     }
    else
     {
      FS_ADD("MOV R0, cc_c2;");
     }
    break;
   //--------------------------------------------------//
   case TE_CC_RASC:
    FS_ADD("MOV R0, fragment.color.primary;");
    break;
   //--------------------------------------------------//
   case TE_CC_RASA:
    FS_ADD("MOV R0, fragment.color.primary.wwww;");
    break;
   //--------------------------------------------------//
   case TE_CC_ONE:
    FS_ADD("MOV R0, one;");
    break;
   //--------------------------------------------------//
   case TE_CC_HALF:
    FS_ADD("MOV R0, half;");
    break;
   //--------------------------------------------------//
   case TE_CC_KONST:
    // tev_konst is in format: 0000 00aa aaac cccc
    switch(tev_konst[gstage] & 0x1f)                   
     {//-----------------------------------------------//
      case TE_KCSEL_K0:
       if(chan == CHAN_D)
        {
         FS_ADD("MOV R0, env_k0_s10;");
        }
       else
        {
         FS_ADD("MOV R0, env_k0;");
        }
       break;
      //-----------------------------------------------//
      case TE_KCSEL_K1:
       if(chan == CHAN_D)
        {
         FS_ADD("MOV R0, env_k1_s10;");
        }
       else
        {
         FS_ADD("MOV R0, env_k1;");
        }
       break;
      //-----------------------------------------------//
      case TE_KCSEL_K2:
       if(chan == CHAN_D)
        {
         FS_ADD("MOV R0, env_k2_s10;");
        }
       else
        {
         FS_ADD("MOV R0, env_k2;");
        }
       break;
      //-----------------------------------------------//
      case TE_KCSEL_K3:
       if(chan == CHAN_D)
        {
         FS_ADD("MOV R0, env_k3_s10;");
        }
       else
        {
         FS_ADD("MOV R0, env_k3;");
        }
       break;
      //-----------------------------------------------//
      case TE_KCSEL_K0_A:
       if(chan == CHAN_D)
        {
         FS_ADD("MOV R0, env_k0_s10.wwww;");
        }
       else
        {
         FS_ADD("MOV R0, env_k0.wwww;");
        }
       break;
      //-----------------------------------------------//
      case TE_KCSEL_K1_A:
       if(chan == CHAN_D)
        {
         FS_ADD("MOV R0, env_k1_s10.wwww;");
        }
       else
        {
         FS_ADD("MOV R0, env_k1.wwww;");
        }
       break;
      //-----------------------------------------------//
      case TE_KCSEL_K3_A:
       if(chan == CHAN_D)
        {
         FS_ADD("MOV R0, env_k3_s10.wwww;");
        }
       else
        {
         FS_ADD("MOV R0, env_k3.wwww;");
        }
       break;
      //-----------------------------------------------//
      case TE_KCSEL_K1_R:
       if(chan == CHAN_D)
        {
         FS_ADD("MOV R0, env_k1_s10.xxxx;");
        }
       else
        {
         FS_ADD("MOV R0, env_k1.xxxx;");
        }
       break;
      //-----------------------------------------------//
      case TE_KCSEL_K3_B:
       if(chan == CHAN_D)
        {
         FS_ADD("MOV R0, env_k3_s10.zzzz;");
        }
       else
        {
         FS_ADD("MOV R0, env_k3.zzzz;");
        }
       break;
      //-----------------------------------------------//
      case TE_KCSEL_1:
       FS_ADD("MOV R0, one;");
       break;
      //-----------------------------------------------//
      case TE_KCSEL_1_4:
       FS_ADD("MOV R0, quart;");
       break;
      //-----------------------------------------------//
      default:
       //fprintf(stderr, "KONST %s", te_kcsel[tev_konst[gstage] & 0x1f]);
       // gpu_exit(0,3);
       break;
      //-----------------------------------------------//
     }
    break;
   //--------------------------------------------------//
   case TE_CC_ZERO:
    FS_ADD("MOV R0, zero;");
    break;
   //--------------------------------------------------//
   default:
    FS_ADD("MOV R0, zero;");
    //fprintf(stderr, "Unsupported input: %s\n", te_color_inputs[input]);
    //gpu_exit(0,4);
    break;
   //--------------------------------------------------//
  }
}

///////////////////////////////////////////////////////////////////////////

static bool fsg_generate_color_and_alpha_chan(char *fs, uint8 input_c, uint8 input_a, uint8 chan)
{
 uint8 input = (input_c << 4) | input_a;

 switch(input)
  {
   case FSG_DUAL_CH(TE_CC_ZERO, TE_CA_ZERO):
    FS_ADD("MOV R0, zero;");
    return true;
 
   case FSG_DUAL_CH(TE_CC_RASC, TE_CA_RASA):
    FS_ADD("MOV R0, fragment.color.primary;");
    //FS_ADD("MOV R0, one;");
    return true;

   default:
    break;
  }

 return false;
}

///////////////////////////////////////////////////////////////////////////

void print_swap_table(uint32 alpha)
{
 uint32 table_entry_ras;
 uint32 table_entry_tex;

 table_entry_ras = te_swap_table[(alpha >> 0) & 0x3];
 table_entry_tex = te_swap_table[(alpha >> 2) & 0x3];

 if (table_entry_ras == 0x4e && table_entry_tex == 0x4e)
  return;

 //syslog(TE,"SWAP TABLE RAS: %d (%02x) TEX: %d (%02x)\n", (alpha >> 0) & 0x3, table_entry_ras, (alpha >> 2) & 0x3, table_entry_tex);
}

///////////////////////////////////////////////////////////////////////////

void fsg_generate_program_source(void)
{
 uint32  i;
 char    *fs;
 uint32 color, alpha;
 char tmpstr[256];

 fs = fsg_program_source;

 cc_used = 0;
 ca_used = 0;

 fsg_icount = 0;
    
 FS_INIT();
 FS_HEADER();

//  if(input_a == TE_CC_C0 || input_b == TE_CC_C0 || input_c == TE_CC_C0 || input_d == TE_CC_C0)
//  {
//      FS_ADD("PARAM cc_c0 = program.local[0];");
//      FS_ADD("PARAM cc_c0 = program.env[0];");
//  }
//  if(input_a == TE_CC_C1 || input_b == TE_CC_C1 || input_c == TE_CC_C1 || input_d == TE_CC_C1)
//  {
//      FS_ADD("PARAM cc_c1 = program.local[1];");
//  }

 //FS_ADD("OPTION ARB_FOG_LINEAR;");

 if(bp_fog==2) {FS_ADD("OPTION ARB_fog_linear;");}
 if(bp_fog==4) {FS_ADD("OPTION ARB_fog_exp;");}
 if(bp_fog==5) {FS_ADD("OPTION ARB_fog_exp2;");}

 FS_ADD("PARAM env_prev = program.env[0];");
 FS_ADD("PARAM env_c0 = program.env[1];");
 FS_ADD("PARAM env_c1 = program.env[2];");
 FS_ADD("PARAM env_c2 = program.env[3];");
 FS_ADD("PARAM env_k0 = program.env[4];");
 FS_ADD("PARAM env_k1 = program.env[5];");
 FS_ADD("PARAM env_k2 = program.env[6];");
 FS_ADD("PARAM env_k3 = program.env[7];");
 FS_ADD("PARAM env_prev_s10 = program.env[8];");
 FS_ADD("PARAM env_c0_s10 = program.env[9];");
 FS_ADD("PARAM env_c1_s10 = program.env[10];");
 FS_ADD("PARAM env_c2_s10 = program.env[11];");
 FS_ADD("PARAM env_k0_s10 = program.env[12];");
 FS_ADD("PARAM env_k1_s10 = program.env[13];");
 FS_ADD("PARAM env_k2_s10 = program.env[14];");
 FS_ADD("PARAM env_k3_s10 = program.env[15];");

 FS_ADD_VAR("PARAM quart = {0.25, 0.25, 0.25, 0.25};");
 FS_ADD_VAR("PARAM half = {0.5, 0.5, 0.5, 0.5};");
 FS_ADD_VAR("PARAM zero = {0, 0, 0, 0};");
 FS_ADD_VAR("PARAM one = {1, 1, 1, 1};");
 FS_ADD_VAR("PARAM c2 = {2, 2, 2, 2};");
 FS_ADD_VAR("PARAM c4 = {4, 4, 4, 4};");
 FS_ADD_VAR("TEMP R0;");
 FS_ADD_VAR("TEMP R1;");
 FS_ADD_VAR("TEMP R2;");
 FS_ADD_VAR("TEMP R3;");
 FS_ADD_VAR("TEMP cc_prev;");
 FS_ADD_VAR("TEMP cc_c0;");
 FS_ADD_VAR("TEMP cc_c1;");
 FS_ADD_VAR("TEMP cc_c2;");

 FS_ADD("MOV cc_prev, env_prev;");
 FS_ADD("MOV cc_c0, env_c0;");
 FS_ADD("MOV cc_c1, env_c1;");
 FS_ADD("MOV cc_c2, env_c2;");

 for(i=0;i<te_active_stages;i++)
  {
   gstage = i;
   color = te_combiner_regs[i * 2 + 0];
   alpha = te_combiner_regs[i * 2 + 1];

   //syslog(TE,"CLR STAGE: %d next: %03x inputs: %06x %s %s %s %s bias: %d sub: %d clamp: %d shift: %d dest: %d\n", gstage, tev_stage[i], color, te_color_inputs[(color >> 12) & 0xf], te_color_inputs[(color >> 8) & 0xf], te_color_inputs[(color >> 4) & 0xf], te_color_inputs[(color >> 0) & 0xf], (color >> 16) & 0x3, (color >> 18) & 0x1, (color >> 19) & 0x1, (color >> 20) & 0x3, (color >> 22) & 0x3);
   //syslog(TE,"ALP STAGE: %d next: %03x inputs: %06x %s %s %s %s bias: %d sub: %d clamp: %d shift: %d dest: %d\n", gstage, tev_stage[i], alpha, te_alpha_inputs[(alpha >> 13) & 0x7], te_alpha_inputs[(alpha >> 10) & 0x7], te_alpha_inputs[(alpha >> 7) & 0x7], te_alpha_inputs[(alpha >> 4) & 0x7], (color >> 16) & 0x3, (color >> 18) & 0x1, (color >> 19) & 0x1, (color >> 20) & 0x3, (color >> 22) & 0x3);
   //syslog(TE,"tev_stage: cc: %d ti: %d tc: %d te: %d\n", (tev_stage[i] >> 7) & 0xf, (tev_stage[i] >> 0) & 0x7, (tev_stage[i] >> 3) & 0x7, (tev_stage[i] >> 6) & 0x1);

   print_swap_table(alpha);

   sprintf(tmpstr, "Stage %d", i);
   FS_ADD_COMMENT(tmpstr);

   if (color != 0)
    {
     // res = xxx + B

     // B
     FS_ADD_COMMENT("channel B");
     if (!fsg_generate_color_and_alpha_chan(fs,(uint8)((color >> 8) & 0xf),(uint8)((alpha >> 10) & 0x7),CHAN_B))
      {
       fsg_generate_color_chan(fs,(uint8)((color >> 8) & 0xf), CHAN_B);
       fsg_generate_alpha_chan(fs,(uint8)((alpha >> 10) & 0x7), CHAN_B);
      }
     FS_ADD("MOV R1, R0;");

     // C
     FS_ADD_COMMENT("channel C");
     if (!fsg_generate_color_and_alpha_chan(fs,(uint8)((color >> 4) & 0xf),(uint8)((alpha >> 7) & 0x7),CHAN_C))
      {
       fsg_generate_color_chan(fs,(uint8)((color >> 4) & 0xf), CHAN_C);
       fsg_generate_alpha_chan(fs,(uint8)((alpha >> 7) & 0x7), CHAN_C);
      }

     // B * C
     FS_ADD("MUL R1, R1, R0;");

     // 1 - C
     FS_ADD("SUB R2, one, R0;");

     // A
     FS_ADD_COMMENT("channel A");
     if(!fsg_generate_color_and_alpha_chan(fs,(uint8)((color >> 12) & 0xf),(uint8)((alpha >> 13) & 0x7), CHAN_A))
      {
       fsg_generate_color_chan(fs,(uint8)((color >> 12) & 0xf), CHAN_A);
       fsg_generate_alpha_chan(fs,(uint8)((alpha >> 13) & 0x7), CHAN_A);
       // A * (1 - C) + B * C
      }
     FS_ADD("MAD R1, R2, R0, R1;");
            
     // D
     FS_ADD_COMMENT("channel D");
     if(!fsg_generate_color_and_alpha_chan(fs,(uint8)((color >> 0) & 0xf),(uint8)((alpha >> 4) & 0x7), CHAN_D))
      {
       fsg_generate_color_chan(fs,(uint8)((color >> 0) & 0xf), CHAN_D);
       fsg_generate_alpha_chan(fs,(uint8)((alpha >> 4) & 0x7), CHAN_D);
      }
     FS_ADD("ADD R1, R1, R0;");

     FS_ADD_COMMENT("Scaling");
     // scale

     switch((color >> 20) & 0x3)
      {
       case 1:
        FS_ADD("MUL R1.xyz, c2, R1;");
        break;
       case 2:
        FS_ADD("MUL R1.xyz, c4, R1;");
        break;
       case 3:
        FS_ADD("MUL R1.xyz, half, R1;");
        break;
      }

     switch((alpha >> 20) & 0x3)
      {
       case 1:
        FS_ADD("MUL R1.w, c2, R1;");
        break;
       case 2:
        FS_ADD("MUL R1.w, c4, R1;");
        break;
       case 3:
        FS_ADD("MUL R1.w, half, R1;");
        break;
      }

     FS_ADD_COMMENT("Clamping");
     // CLAMPING
     if ( ((color >> 19) & 0x1) == ((alpha >> 19) & 0x1))
      {
       // clamp color and alpha
       if((color >> 19) & 0x1)
        {
         FS_ADD("MIN R1, one, R1;");
         FS_ADD("MAX R1, zero, R1;");
        }
      }
     else
      {
       if((color >> 19) & 0x1)
        {
         // clamp color
         FS_ADD("MIN R1.xyz, one, R1;");
         FS_ADD("MAX R1.xyz, zero, R1;");
        }
       if((alpha >> 19) & 0x1)
        {
         // alpha color
         FS_ADD("MIN R1.w, one.w, R1.w;");
         FS_ADD("MAX R1.w, zero.w, R1.w;");
        }
      }

     // Selection of destination register
     cc_used |= 1 << ((color >> 22) & 0x3);
     ca_used |= 1 << ((alpha >> 22) & 0x3);

     FS_ADD_COMMENT("Storing");
     if( ((color >> 22) & 0x3) == ((alpha >> 22) & 0x3))
      {
       switch((color >> 22) & 0x3)
        {
         case 0x0:
          FS_ADD("MOV cc_prev, R1;");
          break;
 
         case 0x1:
          FS_ADD("MOV cc_c0, R1;");
          break;
 
         case 0x2:
          FS_ADD("MOV cc_c1, R1;");
          break;
 
         case 0x3:
          FS_ADD("MOV cc_c2, R1;");
          break;
        }
      }
     else
      {
       switch((color >> 22) & 0x3)
        {
         case 0x0:
          FS_ADD("MOV cc_prev.xyz, R1;");
          break;
 
         case 0x1:
          FS_ADD("MOV cc_c0.xyz, R1;");
          break;

         case 0x2:
          FS_ADD("MOV cc_c1.xyz, R1;");
          break;
 
         case 0x3:
          FS_ADD("MOV cc_c2.xyz, R1;");
          break;
        }

       switch((alpha >> 22) & 0x3)
        {
         case 0x0:
          FS_ADD("MOV cc_prev.w, R1;");
          break;
 
         case 0x1:
          FS_ADD("MOV cc_c0.w, R1;");
          break;
 
         case 0x2:
          FS_ADD("MOV cc_c1.w, R1;");
          break;
  
         case 0x3:
          FS_ADD("MOV cc_c2.w, R1;");
          break;
        }
      }
    }
  }

 FS_ADD("MOV result.color, R1;");

 FS_END();

 if (fsg_icount > fsg_icount_peak)
  fsg_icount_peak = fsg_icount;

// fprintf(stderr, "ICOUNT PEAK: %d (current %d)\n", fsg_icount_peak, fsg_icount);
}

///////////////////////////////////////////////////////////////////////////

void fsg_generate(void)
{
 if (!fsg_cache_check(&fsg_fragment_program))
  {
   char *tmpstr;int result;

   fsg_generate_program_source();

   //MessageBox(NULL,fsg_program_source, "s", MB_OK);

   glGenProgramsARB_Ex(1, &fsg_fragment_program);

   //fprintf(stderr, "glGenProgramsARB generated %d\n", fsg_fragment_program);
        
   glBindProgramARB_Ex(GL_FRAGMENT_PROGRAM_ARB, fsg_fragment_program);
        
   glProgramStringARB_Ex(GL_FRAGMENT_PROGRAM_ARB, GL_PROGRAM_FORMAT_ASCII_ARB, (GLsizei)strlen(fsg_program_source), fsg_program_source);

   tmpstr = (char *)glGetString(GL_PROGRAM_ERROR_STRING_ARB);
   
   glGetIntegerv(GL_PROGRAM_ERROR_POSITION_ARB, &result);
   if (strlen(tmpstr) > 0 || result >= 0)
    {
     //fprintf(stderr, "glProgramStringARB failed (line: %d): '%s'", result, tmpstr);
     fprintf(stderr, fsg_program_source);
     fprintf(stderr, "\nglProgramStringARB failed (line: %d): '%s'", result, tmpstr);
     gpu_exit(0,5);
    }
   fsg_cache_add(fsg_fragment_program);
  }

//auxprintf("fog %d, %d\n",fsg_fragment_program,bp_fog);

 glBindProgramARB_Ex(GL_FRAGMENT_PROGRAM_ARB, fsg_fragment_program);
 glEnable(GL_FRAGMENT_PROGRAM_ARB);
}

///////////////////////////////////////////////////////////////////////////


