/*====================================================================

filename:     main.cpp
project:      GCemu
created:      2004-6-18
mail:		  duddie@walla.com

Copyright (c) 2005 Duddie & Tratax

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

====================================================================*/
#include "config.h"

#include <windows.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "system/types.h"
#include "dol_loader.h"
#include "hardware/memory_interface.h"
#include "hardware/video_interface.h"
#include "hardware/external_interface.h"
#include "hardware/dvd_interface.h"
#include "hardware/pe_interface.h"
#include "hardware/dsp_interface.h"
#include "hardware/processor_interface.h"
#include "hardware/serial_interface.h"
#include "ipl_rom.h"
#include "elf_file.h"
#include "elf_loader.h"
#include "gcm_loader.h"
#include "gcm_wrapper.h"
#include "gcm/GCMutils.h"
#include "gcm/GCMExtras.h"
#include "debug/tracers.h"
#include "debug/wnd_debugger.h"
#include "hardware/memory_card.h"
#include "hardware/dsp_interface.h"
#include "cpu/trx_ppc_cpu.h"
#include "w32_display.h"
#include "version.h"
#include "gdsp_tool.h"
#include "dsp/gdsp_interpreter.h"

#include "plugins/gx_plugin.h"
elf_file *elf = NULL;

gd_globals_t *gdg;


#define SAVE_MEMDUMP 0

typedef struct
{
	char *long_name;
	char *short_name;
} switch_t;


extern uint32 callstack[];
extern uint32 callstack_ptr;

int memlocation = 0;

// sets up all the system values to match given region
//
#define USA 0
#define EUROPE 1
#define JAPAN 2

void set_region(uint32 region)
{
	uint32 tvmode;

	// only europe uses PAL (1)
	switch(region)
	{
	case USA:
		printf("region: USA, NTSC\n");tvmode = 0;break;
	case JAPAN:
		printf("region: JAPAN, NTSC\n");tvmode = 0; break;
	case EUROPE:
		printf("region: EUROPE, PAL\n");tvmode = 1; break;
	}

	// tv mode for OS usage
	mem_write32(0x800000cc, tvmode); // 0 = NTSC, 1 = PAL
}
volatile uint32 dsp_running = 0;
CRITICAL_SECTION xcs;

DWORD WINAPI dsp_thread(LPVOID lpParameter)
{
	dsp_running = 1;
	while(1) 
	{
		if (dsp_running)
		{
		if (!gdsp_run())
		{
			fprintf(stderr, "*** DSP: CRITICAL ERROR ***\n");
			//return 0;
			exit(0);
		}
		}
		Sleep(0);
	};
}

// get extension
static char *get_extension (char *ext, const char *filename)
{
	int i, len;
	len = strlen(filename);
	for(i = len; i>0; i--)
	{
		if(filename[i]=='.')
		{
			strcpy(ext, &filename[i+1]);
			return ext;
		}
	}
	return ".gcm";
}

int main(int argc, char* argv[])
{
	uint32 i;

	printf("GCEMU version: %s\n", version);

	if (argv[1] == NULL)
	{
		printf("USAGE: GCEmu [filename] <options>\n");
		config_help(NULL);
		exit(0);
	}

	// initialise configuration to default and override options based on commandline options
	for(i = 0; i < argc; i++)
	{
		if (argv[i][0] == '-')
		{
			if ( config_variable(argv[i]+1) == false )
			{
				printf("ERROR: unknown switch to me: %s\n", argv[i]);
				exit(1);
			}
		}
    }

	// set up different fpucontrol words for rounding
	_asm
	{
		fnstcw fpucontrol_default
	}
	printf("[CPU] default FPU control word: %x\n", fpucontrol_default);
	fpucontrol_roundzero = fpucontrol_default | 0x0c00;

	// $$$ DEBUG TEST FOR TRIGGERING ASSEMBLY GENERATION $$$$
	_asm
	{
	}


	gcm_init();

	mem_init();
	trx_cpu_init();

	dspi_init();
	ipl_init();
	exi_init();
	di_init();
	pi_init();
	si_init();

	mc_init();

	printf("using: [%s]\n", config_gpuplugin);
	gx_plugin_init(config_gpuplugin);

	GX_Init(gMemory);

	gx_cache_table = GX_CacheGetTable();

#if WITH_FRAGMENT_SHADER
	fsg_init();
#endif


	mem_write32(0x000000f8, 162000000);
	mem_write32(0x000000fc, 486000000);

	mem_write32(0x000000ec, 0x81800000);
	// memory size on retail console 24MB
	mem_write32(0x00000028, 0x01800000);
	mem_write32(0x000000f0, 0x01800000);

	// for demos ArenaHi must be set
	mem_write32(0x00000030, 0x80430000);
	mem_write32(0x00000034, 0x81700000);

	// JTAG present
	mem_write32(0x00000020, 0xe5207c22);
	
	// Latest devkit
	mem_write32(0x0000002c, 0x10000006);

	mem_write32(0x000000e4, 0x0eef);

	char ext[256];
	uint32 entry_point;

	get_extension (ext, argv[1]);

	set_region(config_region);
	if (stricmp(ext, "dol") == 0)
	{
		gcm_open("demos.gcm");
		gcm_load_fst(gMemory);
		 
		if (!dol_load_file(argv[1], gMemory, entry_point))
		{
			syslog_error(MAIN,"cant load file\n");
		}
	}
	else if (stricmp(ext, "elf") == 0)
	{
		gcm_open("demos.gcm");
		gcm_load_fst(gMemory);

		elf = elf_create(gMemory);
		//elf_enabledebug(elf, false);
		elf_enabledebug(elf, true);
		if (!elf_open(elf, argv[1]))
		{
			syslog_error(MAIN,"Cant open: %s\n", argv[1]);
		}
		elf_loadheaders(elf);
		entry_point = elf_getentrypoint(elf);

		//printf("ELF TOP %08x\n", elf_get_top(elf));
		// Set ArenaLo to ELF top
		printf("Setting ArenaLo to %08x\n", elf_getfunctionaddr(elf, "__ArenaLo"));
		mem_write32(0x00000030, elf_getfunctionaddr(elf, "__ArenaLo"));
		printf("Setting ArenaHi to %08x\n", elf_getfunctionaddr(elf, "__ArenaHi"));
		mem_write32(0x00000034, elf_getfunctionaddr(elf, "__ArenaHi"));
	}
	else if (stricmp(ext, "gcm") == 0)
	{
		FILE *in;
		uint32 region;
		if (!gcm_load_file(argv[1], gMemory, entry_point))
		{
			syslog_error(MAIN,"cant load file\n");
		}
		// configure regionmode
		in = fopen(argv[1], "rb");
		switch(GCMGetRegionCode(in)) 
		{
			case GCM_USA_NTSC: region = USA; break;
			case GCM_EUR_PAL: region = EUROPE; break;
			case GCM_JAP_NTSC: region = JAPAN; break;
			default: region = USA;break;
		}
		fclose(in);
		set_region(region);
	}
	else
	{
		syslog_error(MAIN,"Unknown extension\n");
	}
	CPUcurrmode->pc = entry_point;
	CPUcurrmode->lr = 0;

	HWND wnd = w32_init();

	GX_Open(wnd);


#if WITH_REAL_DSP
	gdsp_init();
	dsp->cpu_ram = gMemory;
	dsp->irq_request = dspi_req_dsp_irq;
	gdsp_reset();
	if (!gdsp_load_rom("gc_dsp_rom.bin"))
	{
		syslog_error(MAIN,"Cannot load DSP ROM\n");
	}
#if WITH_DSP_ON_THREAD
	CreateThread(NULL, 0, dsp_thread, 0, 0, NULL /* threadID */);
	InitializeCriticalSection(&xcs);
#endif // WITH_DSP_ON_THREAD
#endif

	if (config_debug)
	{
		// with debugger
		// initalize dsp toolkit
		gdg = gd_init();

		gdg->decode_registers = false;
		gdg->decode_names = false;
		gdg->show_hex = true;
		gdg->show_pc = true;
		gdg->print_tabs = false;
		gdg->ext_separator = '\'';
		dbg_main();
	}
	else
	{
		trx_ppc_run();
	}

#if SAVE_MEMDUMP
	FILE *out;
	out = fopen("memdump.bin", "wb");
	fwrite(gMemory, 1, 24 * 1024 * 1024, out);
	fclose(out);
#endif

	printf("Finished execution\n");
	exit(0);
}


