/*====================================================================

filename:     memory_interface.cpp
project:      GCemu
created:      2004-6-18
mail:		  duddie@walla.com

Copyright (c) 2005 Duddie & Tratax

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

====================================================================*/
#include <stdio.h>
#include "hardware/memory_interface.h"
#include "hardware/hw_io.h"

// for rec cache snooping
#include "cpu/trx_ppc_rec.h"
// for gpu cache snooping
#include "plugins/gx_plugin.h"
int mem_init()
{
	gMemory = (uint8*)malloc(24*1024*1024);
	if(gMemory == NULL)
	{
		printf("failed to allocate 24Mbytes of main ram\n");
		exit(0);
	}
	memset(gMemory, 0, 24*1024*1024);
	return 0;
}

// for debugging .. read memory without touching for example IO registers and reporting illegal areas
bool mem_peek32(uint32 address, uint32 *data)
{
	uint32 val;
	if(address < 0xc8000000)
	{
		// RAM address
		_asm
		{
			mov eax, address
			mov edx, gMemory
			and eax, MEM_MASK32MB
			mov eax, [edx+eax]
			bswap eax
			mov val, eax
		}
		*data = val;
		return true;
	}
	return false;
}

// for debugging .. read memory without touching for example IO registers and reporting illegal areas
bool mem_peek8(uint32 address, uint8 *data)
{
	if(address < 0xc8000000)
	{
		*data = gMemory[address & MEM_MASK32MB];
		return true;
	}
	return false;
}

// normal memory routines

uint8 mem_read8(uint32 address)
{
	if(address < 0xc8000000)
	{
		// virtual memory (32 MB area)
		if(address >= 0x7e000000 && address <= 0x7fffffff)
		{
			return virtualmemory[address & MEM_MASK32MB];	
		}
		else
		{
			// RAM address
			return gMemory[address & MEM_MASK32MB];
		}
	}
	else
	{	
		if(address >= 0xe0000000)
		{             
			return lockedcache[address & LOCKEDCACHE_MASK];
		}
		else if ((address & 0xff000000) == 0xcc000000)
		{
			return mem_read_io8(address);
		}
	}
	printf("mem_read8: unhandled address: %x\n", address);
	exit(0);
}
uint16 mem_read16(uint32 address)
{
	uint16 val;
	if(address < 0xc8000000)
	{
		// virtual memory (32 MB area)
		if(address >= 0x7e000000 && address <= 0x7fffffff)
		{
			_asm
			{
				mov eax, address
				mov edx, virtualmemory
				and eax, MEM_MASK32MB
				mov ax, [edx+eax]
				xchg al, ah
				mov val, ax
			}
		}
		else
		{
			// RAM address
			_asm
			{
				mov eax, address
				mov edx, gMemory
				and eax, MEM_MASK32MB
				mov ax, [edx+eax]
				xchg al, ah
				mov val, ax
			}
		}
		return val;
	}
	else
	{	
		if(address >= 0xe0000000)
		{
			_asm
			{
				mov eax, address
				mov edx, pLockedcache
				and eax, LOCKEDCACHE_MASK
				mov ax, [edx+eax]
				xchg al, ah
				mov val, ax
			}
			return val;
		}
		else if ((address & 0xff000000) == 0xcc000000)
		{
			return mem_read_io16(address);
		}
	}
	printf("mem_read16: unhandled address: %x\n", address);
	exit(0);
}
uint32 mem_iread(uint32 address)
{
	uint32 val;
	if(address < 0xc8000000)
	{
		// virtual memory (32 MB area)
		if(address >= 0x7e000000 && address <= 0x7fffffff)
		{
			_asm
			{
				mov eax, address
				mov edx, virtualmemory
				and eax, MEM_MASK32MB
				mov eax, [edx+eax]
				bswap eax
				mov val, eax
			}
		}
		else
		{
			// RAM address
			_asm
			{
				mov eax, address
				mov edx, gMemory
				and eax, MEM_MASK32MB
				mov eax, [edx+eax]
				bswap eax
				mov val, eax
			}
		}
		return val;
	}
	else
	{	
		if(address >= 0xe0000000)
		{
			_asm
			{
				mov eax, address
				mov edx, pLockedcache
				and eax, LOCKEDCACHE_MASK
				mov eax, [edx+eax]
				bswap eax
				mov val, eax
			}
			return val;
		}
		printf("instruction read from IO area @ PC: %x\n", CPUcurrmode->pc);
		exit(0);
	}
	printf("mem_iread: unhandled address: %x\n", address);
	exit(0);
}

uint32 mem_read32(uint32 address)
{
	uint32 val;
	if(address < 0xc8000000)
	{
		// virtual memory (32 MB area)
		if(address >= 0x7e000000 && address <= 0x7fffffff)
		{
			_asm
			{
				mov eax, address
				mov edx, virtualmemory
				and eax, MEM_MASK32MB
				mov eax, [edx+eax]
				bswap eax
				mov val, eax
			}
		}
		else
		{
			// RAM address
			_asm
			{
				mov eax, address
				mov edx, gMemory
				and eax, MEM_MASK32MB
				mov eax, [edx+eax]
				bswap eax
				mov val, eax
			}
		}
		return val;
	}
	else
	{	
		if(address >= 0xe0000000)
		{
			_asm
			{
				mov eax, address
				mov edx, pLockedcache
				and eax, LOCKEDCACHE_MASK
				mov eax, [edx+eax]
				bswap eax
				mov val, eax
			}
			return val;
		}
		else if ((address & 0xff000000) == 0xcc000000)
		{
			return mem_read_io32(address);
		}
	}
	printf("mem_read32: unhandled address: %x\n", address);
	exit(0);
}
void mem_write8(uint32 address, uint8 val)
{
	if(address < 0xc8000000)
	{
		// virtual memory (32 MB area)
		if(address >= 0x7e000000 && address <= 0x7fffffff)
		{
			virtualmemory[address & MEM_MASK32MB] = val;
		}
		else
		{
			// RAM address
			gMemory[address & MEM_MASK32MB] = val;
			rec_cache_snoop(address&MEM_MASK32MB);
			GX_CacheSnoop(address&MEM_MASK32MB);
		}
	}
	else
	{
		if(address >= 0xe0000000)
		{
			lockedcache[address & LOCKEDCACHE_MASK] = val;
		}
		else if ((address & 0xff000000) == 0xcc000000)
		{
			return mem_write_io8(address, val);
		}
	}
}
void mem_write16(uint32 address, uint16 val)
{
	if(address < 0xc8000000)
	{
		// virtual memory (32 MB area)
		if(address >= 0x7e000000 && address <= 0x7fffffff)
		{
			_asm
			{
				mov ax, val
				xchg al, ah
				mov ecx, address
				mov edx, virtualmemory
				and ecx, MEM_MASK32MB
				mov [edx+ecx], ax
			}
		}
		else
		{
			_asm
			{
				mov ax, val
				xchg al, ah
				mov ecx, address
				mov edx, gMemory
				and ecx, MEM_MASK32MB
				mov [edx+ecx], ax
			}
			rec_cache_snoop(address&MEM_MASK32MB);
			GX_CacheSnoop(address&MEM_MASK32MB);
		}
	}
	else
	{
		if(address >= 0xe0000000)
		{
			_asm
			{
				mov ax, val
				xchg al, ah
				mov ecx, address
				mov edx, pLockedcache
				and ecx, LOCKEDCACHE_MASK
				mov [edx+ecx], ax
			}
		}
		else if ((address & 0xff000000) == 0xcc000000)
		{
			return mem_write_io16(address, val);
		}
	}
}
void mem_write32(uint32 address, uint32 val)
{
	if(address < 0xc8000000)
	{
		// virtual memory (32 MB area)
		if(address >= 0x7e000000 && address <= 0x7fffffff)
		{
			_asm
			{
				mov eax, val
				bswap eax
				mov ecx, address
				mov edx, virtualmemory
				and ecx, 0x01ffffff
				mov [edx+ecx], eax
			}
		}
		else
		{
			_asm
			{
				mov eax, val
				bswap eax
				mov ecx, address
				mov edx, gMemory
				and ecx, 0x01ffffff
				mov [edx+ecx], eax
			}
			rec_cache_snoop(address & MEM_MASK32MB);
			GX_CacheSnoop(address & MEM_MASK32MB);
		}
	}
	else
	{
		if(address >= 0xe0000000)
		{
			_asm
			{
				mov eax, val
				bswap eax
				mov ecx, address
				mov edx, pLockedcache
				and ecx, LOCKEDCACHE_MASK
				mov [edx+ecx], eax
			}
		}
		else if ((address & 0xff000000) == 0xcc000000)
		{
			return mem_write_io32(address, val);
		}
	}
}