/*====================================================================

filename:     hw_io.cpp
project:      GCemu
created:      2004-6-18
mail:		  duddie@walla.com

Copyright (c) 2005 Duddie & Tratax

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

====================================================================*/
#include "hardware/hw_io.h"
#include <stdio.h>
#define PPC_PC	CPUcurrmode->pc

#define WITH_WPAR 1


extern void gx_write_fifo32(uint32 data);
extern void gx_write_fifo16(uint16 data);
extern void gx_write_fifo8(uint8 data);

//==============================================================================
// WPAR engine
//
// Write gathering FIFO. combines writes to WPAR monitored address into 32byte bursts of data
//
// Only gets configured for register 0x0c008000 which is going to the chipsets command processor
// then the command processor either passes it on to the GPU or dumps it to memory
// 

uint8 wpar_buffer[64]; // 32bytes + overflow
uint32 wpar_index;
uint32 wpar_address;
uint32 wpar_hasdata;

void wpar_set(void)
{
	// initialize WPAR to new address and trash any data in buffer
	wpar_address = CPUcurrmode->spr[PPC_WPAR] & 0xffffffe0;
	wpar_index = 0;
	wpar_hasdata = 0;

	if(wpar_address != 0x0c008000)
	{
		syslog_error(MAIN, "WPAR set to: %x which isnt GX FIFO!\n", wpar_address);
	}
	else
	{
		printf("WPAR set to: %x\n", wpar_address);
	}
}

void flush_wpar(void)
{
	// write data line from WPAR cache in a 256bit burst to the command processor
	cp_write_256(wpar_buffer);

	wpar_index -=32;
	if(wpar_index > 0)
	{
		memcpy(&wpar_buffer[0], &wpar_buffer[32], wpar_index); 
	}
	else
	{
		wpar_hasdata = 0;
	}
}

void wpar_write8(uint8 data)
{
//	syslog(GX,"%08x WPAR: %2.2x\n", CPUcurrmode->pc, data);
	wpar_hasdata = 1;
	wpar_buffer[wpar_index] = data;
	wpar_index++;
	if(wpar_index >=32)
	{
		flush_wpar();
	}
}

void wpar_write16(uint16 data)
{
	uint16 val = data;

//	syslog(GX,"%08x WPAR: %4.4x\n", CPUcurrmode->pc, data);
	wpar_hasdata = 1;
	_asm
	{
		mov ax, val
		xchg al, ah // we need to swap since wpar data can stream to memory and this is last moment we know the write 'size'
		mov ecx, wpar_index
		mov word ptr wpar_buffer[ecx], ax
	}

	wpar_index += 2;
	if(wpar_index >=32)
	{
		flush_wpar();
	}
}

void wpar_write32(uint32 data)
{
	uint32 val = data;

//	syslog(GX,"%08x WPAR: %8.8x\n", CPUcurrmode->pc, data);
	wpar_hasdata = 1;
	_asm
	{
		mov eax, val
		bswap eax // we need to swap since wpar data can stream to memory and this is last moment we know the write 'size'
		mov ecx, wpar_index
		mov dword ptr wpar_buffer[ecx], eax
	}

	wpar_index += 4;
	if(wpar_index >=32)
	{
		flush_wpar();
	}
}

void mem_write_io8(uint32 addr, uint8 data)
{
	int size=1;

#if WITH_WPAR
	// WPAR (physical address .. 0xcc00800x becomes 0x0c008000) 
	if((addr & 0x0ffffff0) == wpar_address)
	{
		return wpar_write8(data);
	}
#endif
	switch((addr>>8) & 0xff)
	{
	case 0x00:
		// command processor
		cp_write_register8(addr, data);
		syslog_error(IO_CORE,"%08x HWW CP_%04x (%08x.%x)\n", PPC_PC, addr & 0xffff, data, size);
		break;
	case 0x10:
		// pixel engine
		pe_write_register8(addr, data);
		syslog_error(IO_CORE,"HWW PE_%04x (%08x.%x)\n", addr & 0xffff, data, size);
		break;
	case 0x20:
		vi_write_register8(addr, data);
		syslog_error(IO_CORE,"HWW VI_%04x (%08x.%x)\n", addr & 0xffff, data, size);
		break;
	case 0x30:
		pi_write_register8(addr, data);
		syslog_error(IO_CORE,"HWW PI_%04x (%08x.%x)\n", addr & 0xffff, data, size);
		break;
	case 0x40:
		syslog_error(IO_CORE,"HWW MI_%04x (%08x.%x)\n", addr & 0xffff, data, size);
		break;
	case 0x50:
		dspi_write_register8(addr, data);
		syslog_error(IO_CORE,"%08x HWW DSPI_%04x (%08x.%x)\n", PPC_PC, addr & 0xffff, data, size);
		break;
	case 0x60:
		di_write_register8(addr, data);
		syslog_error(IO_CORE,"HWW DVDI_%04x (%08x.%x)\n", addr & 0xffff, data, size);
		break;
	case 0x64:
		//si_write_register8(addr, data);
		syslog_error(IO_CORE,"HWW SI_%04x (%08x.%x)\n", addr & 0xffff, data, size);
		break;
	case 0x68:
		exi_write_register8(addr, data);
		syslog_error(IO_CORE,"%08x HWW EXI_%04x (%08x.%x)\n", PPC_PC, addr & 0xffff, data, size);
		break;
	case 0x6c:
		ai_write_register8(addr, data);
		syslog_error(IO_CORE,"%08x HWW AI_%04x (%08x.%x)\n", PPC_PC, addr & 0xffff, data, size);
		break;
	case 0x80:
		gx_write_fifo8(data);
		// syslog(IO_CORE,"HWW GX_%04x (%08x.%x)\n", addr & 0xffff, data, size);
		break;
	default:
		syslog_error(IO_CORE,"no one is responsible for address %08x (write: %08x from %08x)\n", addr, data, PPC_PC);
	}
}

void mem_write_io16(uint32 addr, uint16 data)
{
	int size=2;

#if WITH_WPAR
	// WPAR (physical address .. 0xcc00800x becomes 0x0c008000) 
	if((addr & 0x0ffffff0) == wpar_address)
	{
		return wpar_write16(data);
	}
#endif

	switch((addr>>8) & 0xff)
	{
	case 0x00:
		// command processor
		cp_write_register16(addr, data);
		syslog(IO_CORE,"%08x HWW CP_%04x (%08x.%x)\n", PPC_PC, addr & 0xffff, data, size);
		break;
	case 0x10:
		// pixel engine
		pe_write_register16(addr, data);
		syslog(IO_CORE,"HWW PE_%04x (%08x.%x)\n", addr & 0xffff, data, size);
		break;
	case 0x20:
		vi_write_register16(addr, data);
		syslog(IO_CORE,"HWW VI_%04x (%08x.%x)\n", addr & 0xffff, data, size);
		break;
	case 0x30:
		pi_write_register16(addr, data);
		syslog_error(IO_CORE,"HWW PI_%04x (%08x.%x)\n", addr & 0xffff, data, size);
		break;
	case 0x40:
		// we totally ignore the memory protection services here for now
		//syslog_error(IO_CORE,"HWW MI_%04x (%08x.%x)\n", addr & 0xffff, data, size);
		break;
	case 0x50:
		dspi_write_register16(addr, data);
		syslog(IO_CORE,"%08x HWW DSPI_%04x (%08x.%x)\n", PPC_PC, addr & 0xffff, data, size);
		break;
	case 0x60:
		di_write_register16(addr, data);
		syslog(IO_CORE,"HWW DVDI_%04x (%08x.%x)\n", addr & 0xffff, data, size);
		break;
	case 0x64:
		//si_write_register16(addr, data);
		syslog(IO_CORE,"HWW SI_%04x (%08x.%x)\n", addr & 0xffff, data, size);
		break;
	case 0x68:
		exi_write_register16(addr, data);
		syslog(IO_CORE,"%08x HWW EXI_%04x (%08x.%x)\n", PPC_PC, addr & 0xffff, data, size);
		break;
	case 0x6c:
		ai_write_register16(addr, data);
		syslog(IO_CORE,"%08x HWW AI_%04x (%08x.%x)\n", PPC_PC, addr & 0xffff, data, size);
		break;
	case 0x80:
		gx_write_fifo16(data);
		// syslog(IO_CORE,"HWW GX_%04x (%08x.%x)\n", addr & 0xffff, data, size);
		break;
	default:
		syslog_error(IO_CORE,"no one is responsible for address %08x (write: %08x from %08x)\n", addr, data, PPC_PC);
	}
}

void mem_write_io32(uint32 addr, uint32 data)
{
	int size=4;

#if WITH_WPAR
	// WPAR (physical address .. 0xcc00800x becomes 0x0c008000) 
	if((addr & 0x0ffffff0) == wpar_address)
	{
		return wpar_write32(data);
	}
#endif

	switch((addr>>8) & 0xff)
	{
	case 0x00:
		// command processor
		cp_write_register32(addr, data);
		syslog_error(IO_CORE,"%08x HWW CP_%04x (%08x.%x)\n", PPC_PC, addr & 0xffff, data, size);
		break;
	case 0x10:
		// pixel engine
		pe_write_register32(addr, data);
		syslog_error(IO_CORE,"HWW PE_%04x (%08x.%x)\n", addr & 0xffff, data, size);
		break;
	case 0x20:
		vi_write_register32(addr, data);
		syslog(IO_CORE,"HWW VI_%04x (%08x.%x)\n", addr & 0xffff, data, size);
		break;
	case 0x30:
		pi_write_register32(addr, data);
		syslog(IO_CORE,"HWW PI_%04x (%08x.%x)\n", addr & 0xffff, data, size);
		break;
	case 0x40:
		// we totally ignore the memory protection services here for now
		//syslog_error(IO_CORE,"HWW MI_%04x (%08x.%x)\n", addr & 0xffff, data, size);
		break;
	case 0x50:
		dspi_write_register32(addr, data);
		syslog(IO_CORE,"%08x HWW DSPI_%04x (%08x.%x)\n", PPC_PC, addr & 0xffff, data, size);
		break;
	case 0x60:
		di_write_register32(addr, data);
		syslog(IO_CORE,"HWW DVDI_%04x (%08x.%x)\n", addr & 0xffff, data, size);
		break;
	case 0x64:
		si_write_register32(addr, data);
		syslog(IO_CORE,"HWW SI_%04x (%08x.%x)\n", addr & 0xffff, data, size);
		break;
	case 0x68:
		exi_write_register32(addr, data);
		syslog(IO_CORE,"%08x HWW EXI_%04x (%08x.%x)\n", PPC_PC, addr & 0xffff, data, size);
		break;
	case 0x6c:
		ai_write_register32(addr, data);
		syslog(IO_CORE,"%08x HWW AI_%04x (%08x.%x)\n", PPC_PC, addr & 0xffff, data, size);
		break;
	case 0x80:
		gx_write_fifo32(data);
		// syslog(IO_CORE,"HWW GX_%04x (%08x.%x)\n", addr & 0xffff, data, size);
		break;
	default:
		syslog_error(IO_CORE,"no one is responsible for address %08x (write: %08x from %08x)\n", addr, data, PPC_PC);
	}
}

uint8 mem_read_io8(uint32 addr)
{
	uint8 data;
	uint32 size=1;
	switch((addr >> 8) & 0xff)
	{
	case 0x00:
		data = cp_read_register8(addr);
		syslog_error(IO_CORE,"%08x HWR CP_%04x (%08x.%x)\n", PPC_PC, addr & 0xffff, data, size);
		break;
	case 0x10:
		data = pe_read_register8(addr);
		syslog_error(IO_CORE,"%08x HWR PE_%04x (%08x.%x)\n", PPC_PC, addr & 0xffff, data, size);
		break;
	case 0x20:
		data = vi_read_register8(addr);
		syslog_error(IO_CORE,"HWR VI_%04x (%08x.%x)\n", addr & 0xffff, data, size);
		break;
	case 0x30:
		data = pi_read_register8(addr);
		syslog_error(IO_CORE,"%08x HWR PI_%04x (%08x.%x)\n", PPC_PC, addr & 0xffff, data, size);
		break;
	case 0x40:
		data = 0x0;
		syslog_error(IO_CORE,"HWR MI_%04x (%08x.%x)\n", addr & 0xffff, data, size);
		break;
	case 0x50:
		data = dspi_read_register8(addr);
		syslog(IO_CORE,"%08x HWR DSPI_%04x (%08x.%x)\n", PPC_PC, addr & 0xffff, data, size);
		break;
	case 0x60:
		data = di_read_register8(addr);
		syslog_error(IO_CORE,"%08x HWR DVDI_%04x (%08x.%x)\n", PPC_PC, addr & 0xffff, data, size);
		break;
	case 0x64:
		//data = si_read_register8(addr);
		syslog_error(IO_CORE,"HWR SI_%04x (%08x.%x)\n", addr & 0xffff, data, size);
		break;
	case 0x68:
		data = exi_read_register8(addr);
		syslog_error(IO_CORE,"%08x HWR EXI_%04x (%08x.%x)\n", PPC_PC, addr & 0xffff, data, size);
		break;
	case 0x6c:
		data = ai_read_register8(addr);
		syslog_error(IO_CORE,"%08x HWR AI_%04x (%08x.%x)\n", PPC_PC, addr & 0xffff, data, size);
		break;
	case 0x80:
		data = 0x0;
		syslog_error(IO_CORE,"HWR GX_%04x (%08x.%x)\n", addr & 0xffff, data, size);
		break;
	default:
		syslog_error(IO_CORE,"no one is responsible for address %08x (read from %08x)\n", addr, PPC_PC);
	}
	return data;
}

uint16 mem_read_io16(uint32 addr)
{
	uint16 data;
	uint32 size=2;
	switch((addr >> 8) & 0xff)
	{
	case 0x00:
		data = cp_read_register16(addr);
		syslog(IO_CORE,"%08x HWR CP_%04x (%08x.%x)\n", PPC_PC, addr & 0xffff, data, size);
		break;
	case 0x10:
		data = pe_read_register16(addr);
		syslog(IO_CORE,"%08x HWR PE_%04x (%08x.%x)\n", PPC_PC, addr & 0xffff, data, size);
		break;
	case 0x20:
		data = vi_read_register16(addr);
		syslog(IO_CORE,"HWR VI_%04x (%08x.%x)\n", addr & 0xffff, data, size);
		break;
	case 0x30:
		data = pi_read_register16(addr);
		syslog(IO_CORE,"%08x HWR PI_%04x (%08x.%x)\n", PPC_PC, addr & 0xffff, data, size);
		break;
	case 0x40:
		data = 0x0;
		syslog(IO_CORE,"HWR MI_%04x (%08x.%x)\n", addr & 0xffff, data, size);
		break;
	case 0x50:
		data = dspi_read_register16(addr);
		syslog(IO_CORE,"%08x HWR DSPI_%04x (%08x.%x)\n", PPC_PC, addr & 0xffff, data, size);
		break;
	case 0x60:
		data = di_read_register16(addr);
		syslog(IO_CORE,"%08x HWR DVDI_%04x (%08x.%x)\n", PPC_PC, addr & 0xffff, data, size);
		break;
	case 0x64:
		//data = si_read_register16(addr);
		syslog(IO_CORE,"HWR SI_%04x (%08x.%x)\n", addr & 0xffff, data, size);
		break;
	case 0x68:
		data = exi_read_register16(addr);
		syslog(IO_CORE,"%08x HWR EXI_%04x (%08x.%x)\n", PPC_PC, addr & 0xffff, data, size);
		break;
	case 0x6c:
		data = ai_read_register16(addr);
		syslog(IO_CORE,"%08x HWR AI_%04x (%08x.%x)\n", PPC_PC, addr & 0xffff, data, size);
		break;
	case 0x80:
		data = 0x0;
		syslog(IO_CORE,"HWR GX_%04x (%08x.%x)\n", addr & 0xffff, data, size);
		break;
	default:
		syslog_error(IO_CORE,"no one is responsible for address %08x (read from %08x)\n", addr, PPC_PC);
	}
	return data;
}

uint32 mem_read_io32(uint32 addr)
{
	uint32 data;
	uint32 size=4;
	switch((addr >> 8) & 0xff)
	{
	case 0x00:
		data = cp_read_register32(addr);
		syslog(IO_CORE,"%08x HWR CP_%04x (%08x.%x)\n", PPC_PC, addr & 0xffff, data, size);
		break;
	case 0x10:
		data = pe_read_register32(addr);
		syslog(IO_CORE,"%08x HWR PE_%04x (%08x.%x)\n", PPC_PC, addr & 0xffff, data, size);
		break;
	case 0x20:
		data = vi_read_register32(addr);
		syslog(IO_CORE,"HWR VI_%04x (%08x.%x)\n", addr & 0xffff, data, size);
		break;
	case 0x30:
		data = pi_read_register32(addr);
		syslog(IO_CORE,"%08x HWR PI_%04x (%08x.%x)\n", PPC_PC, addr & 0xffff, data, size);
		break;
	case 0x40:
		data = 0x0;
		// we totally ignore the memory protection services here for now
		//syslog_error(IO_CORE,"HWR MI_%04x (%08x.%x)\n", addr & 0xffff, data, size);
		break;
	case 0x50:
		data = dspi_read_register32(addr);
		syslog(IO_CORE,"%08x HWR DSPI_%04x (%08x.%x)\n", PPC_PC, addr & 0xffff, data, size);
		break;
	case 0x60:
		data = di_read_register32(addr);
		syslog(IO_CORE,"%08x HWR DVDI_%04x (%08x.%x)\n", PPC_PC, addr & 0xffff, data, size);
		break;
	case 0x64:
		data = si_read_register32(addr);
		syslog(IO_CORE,"HWR SI_%04x (%08x.%x)\n", addr & 0xffff, data, size);
		break;
	case 0x68:
		data = exi_read_register32(addr);
		syslog(IO_CORE,"%08x HWR EXI_%04x (%08x.%x)\n", PPC_PC, addr & 0xffff, data, size);
		break;
	case 0x6c:
		data = ai_read_register32(addr);
		syslog(IO_CORE,"%08x HWR AI_%04x (%08x.%x)\n", PPC_PC, addr & 0xffff, data, size);
		break;
	case 0x80:
		data = 0x0;
		syslog(IO_CORE,"HWR GX_%04x (%08x.%x)\n", addr & 0xffff, data, size);
		break;
	default:
		syslog_error(IO_CORE,"no one is responsible for address %08x (read from %08x)\n", addr, PPC_PC);
	}
	return data;
}

static inline int io_mem_write64(uint32 addr, uint64 data)
{
/*
	if ((addr & 0xff000000) == 0xcc000000)
	{
		if((addr & 0xffffff00) == 0xcc008000)
		{
			syslog(IO_CORE,"HWR GX64_%04x\n", addr);
			io_mem_write(addr, (uint32)(data >> 32), 4);
			io_mem_write(addr, (uint32) data, 4);
			break;
		}
	}
*/
	syslog_error(IO_CORE,"IO: no one is responsible for address %08x (write64: %016q from %08x)\n", addr, data, PPC_PC);
//	return IO_MEM_ACCESS_FATAL;
}

static inline int io_mem_read64(uint32 addr, uint64 &data)
{
	syslog_error(IO_CORE,"no one is responsible for address %08x (read64 from %08x)\n", addr, PPC_PC);
//	return IO_MEM_ACCESS_FATAL;
}
