/*====================================================================

filename:     external_interface.cpp
project:      GCemu
created:      2004-6-18
mail:		  duddie@walla.com

Copyright (c) 2005 Duddie & Tratax

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

====================================================================*/
#include <stdio.h>
#include "system/types.h"

#include "external_interface.h"
#include "hardware/hw_io.h"
#include "ipl_rom.h"
#include "hardware/memory_interface.h"
#include "memory_card.h"
#include "external_interface_defines.h"

void exi_run(int ch);
void exi_macronix_run(void);
void exi_memcard_run(uint32 card);

char exi_regs[EXI_REGS_SIZE];

uint32 exi_mx_addr;

#define WITH_MEMCARD	1

void exi_init(void)
{
	int i;
	for (i = 0 ; i < EXI_REGS_SIZE ; i++)
	{
		exi_regs[i] = 0;
	}
	EXI_WRITE_UINT32(EXI_REG_CPR1, (EXI_EXTINTMASK | EXI_TCINTMASK | EXI_EXIINTMASK));
}

void exi_close(void)
{
}
uint32	exi_ch1_cmd_ptr;
uint32	exi_ch1_cmd_count;
uint8	exi_ch1_cmd[10];

bool mc_avail = false;

void exi_write_register8(uint32 addr, uint8 data)
{
	uint32	temp32;
	switch(addr & 0xff)
	{
	case EXI_REG_CPR1:
		syslog_error(EXI,"Boom... we didnt expect it :)\n");
		break;
	default:
		break;
	}

	EXI_WRITE_UINT8(addr & 0xff, data);

	switch(addr & 0xff)
	{
	case EXI_REG_CPR1:
		data = EXI_READ_UINT32(EXI_REG_CPR1);
		if ((data & (1 << 12)) == 0)
		{
			exi_ch1_cmd_ptr = 0;
			exi_ch1_cmd_count = 0;
		}

//		if (data == 0)
//			data |= (1 << 12);// | (1 << 7) | (1 << 11) | (1 << 3));
//		data |= (1 << 12) | (1 << 7) | (1 << 11) | (1 << 1);
//		data |= ~(EXI_EXTINTMASK | EXI_TCINTMASK | EXI_EXIINTMASK);
//		data |= (1 << 12);
//		data |= (1 << 11);
//		data = 0;
		//if (mc_avail)
#if WITH_MEMCARD
		data |= (1 << 12);
#endif
		EXI_WRITE_UINT32(EXI_REG_CPR1, data);
		break;
	case EXI_REG_CPR0:
	case EXI_REG_CPR2:
		temp32 = EXI_READ_UINT32(addr & 0xff);
		//temp32 |= (1 << 12);	// device present on given channel
		EXI_WRITE_UINT32(addr & 0xff, temp32);
		if (data == 0)
			exi_mx_addr = 0;
		break;
	case EXI_REG_CR0:
	case EXI_REG_CR1:
	case EXI_REG_CR2:
		exi_run(((addr & 0xff) - EXI_REG_CR0)/ 0x14);
		break;
	default:
		break;
	}
}

void exi_write_register16(uint32 addr, uint16 data)
{
	uint32	temp32;
	switch(addr & 0xff)
	{
	case EXI_REG_CPR1:
		syslog_error(EXI,"Boom... we didnt expect it :)\n");
		break;
	default:
		break;
	}

	EXI_WRITE_UINT16(addr & 0xff, data);

	switch(addr & 0xff)
	{
	case EXI_REG_CPR1:
		data = EXI_READ_UINT32(EXI_REG_CPR1);
		if ((data & (1 << 12)) == 0)
		{
			exi_ch1_cmd_ptr = 0;
			exi_ch1_cmd_count = 0;
		}

//		if (data == 0)
//			data |= (1 << 12);// | (1 << 7) | (1 << 11) | (1 << 3));
//		data |= (1 << 12) | (1 << 7) | (1 << 11) | (1 << 1);
//		data |= ~(EXI_EXTINTMASK | EXI_TCINTMASK | EXI_EXIINTMASK);
//		data |= (1 << 12);
//		data |= (1 << 11);
//		data = 0;
		//if (mc_avail)
#if WITH_MEMCARD
		data |= (1 << 12);
#endif
		EXI_WRITE_UINT32(EXI_REG_CPR1, data);
		break;
	case EXI_REG_CPR0:
	case EXI_REG_CPR2:
		temp32 = EXI_READ_UINT32(addr & 0xff);
		//temp32 |= (1 << 12);	// device present on given channel
		EXI_WRITE_UINT32(addr & 0xff, temp32);
		if (data == 0)
			exi_mx_addr = 0;
		break;
	case EXI_REG_CR0:
	case EXI_REG_CR1:
	case EXI_REG_CR2:
		exi_run(((addr & 0xff) - EXI_REG_CR0)/ 0x14);
		break;
	default:
		break;
	}
}

void exi_write_register32(uint32 addr, uint32 data)
{
	uint32	temp32;
	switch(addr & 0xff)
	{
	case EXI_REG_CPR1:
		temp32 = EXI_READ_UINT32(EXI_REG_CPR1);
		temp32 &= (EXI_EXTINT | EXI_TCINT | EXI_EXIINT);
		temp32 &= ~(data & (EXI_EXTINT | EXI_TCINT | EXI_EXIINT));
		data &= ~(EXI_EXTINT | EXI_TCINT | EXI_EXIINT);
		data |= temp32;
		break;
	default:
		break;
	}

	EXI_WRITE_UINT32(addr & 0xff, data);

	switch(addr & 0xff)
	{
	case EXI_REG_CPR1:
		data = EXI_READ_UINT32(EXI_REG_CPR1);
		if ((data & (1 << 12)) == 0)
		{
			exi_ch1_cmd_ptr = 0;
			exi_ch1_cmd_count = 0;
		}

//		if (data == 0)
//			data |= (1 << 12);// | (1 << 7) | (1 << 11) | (1 << 3));
//		data |= (1 << 12) | (1 << 7) | (1 << 11) | (1 << 1);
//		data |= ~(EXI_EXTINTMASK | EXI_TCINTMASK | EXI_EXIINTMASK);
//		data |= (1 << 12);
//		data |= (1 << 11);
//		data = 0;
		//if (mc_avail)
#if WITH_MEMCARD
		data |= (1 << 12);
#endif
		EXI_WRITE_UINT32(EXI_REG_CPR1, data);
		break;
	case EXI_REG_CPR0:
	case EXI_REG_CPR2:
		temp32 = EXI_READ_UINT32(addr & 0xff);
		//temp32 |= (1 << 12);	// device present on given channel
		EXI_WRITE_UINT32(addr & 0xff, temp32);
		if (data == 0)
			exi_mx_addr = 0;
		break;
	case EXI_REG_CR0:
	case EXI_REG_CR1:
	case EXI_REG_CR2:
		exi_run(((addr & 0xff) - EXI_REG_CR0)/ 0x14);
		break;
	default:
		break;
	}
}

uint8 exi_read_register8(uint32 addr)
{
	uint8 data;

	data = EXI_READ_UINT8(addr & 0xff);

	switch(addr & 0xff)
	{
	case EXI_REG_CPR0:
		//data |= (1 << 12);
		break;
	case EXI_REG_CPR1:
		//data |= ((1 << 12) | (1 << 7) | (1 << 11) | (1 << 1));
//		data |= (1 << 12) | (1 << 7);
#if !WITH_MEMCARD
		data = 0;
#endif
		break;
	case EXI_REG_CPR2:
		break;
	default:
		break;
	}
	return data;
}

uint16 exi_read_register16(uint32 addr)
{
	uint16 data;

	data = EXI_READ_UINT16(addr & 0xff);

	switch(addr & 0xff)
	{
	case EXI_REG_CPR0:
		//data |= (1 << 12);
		break;
	case EXI_REG_CPR1:
		//data |= ((1 << 12) | (1 << 7) | (1 << 11) | (1 << 1));
//		data |= (1 << 12) | (1 << 7);
#if !WITH_MEMCARD
		data = 0;
#endif
		break;
	case EXI_REG_CPR2:
		break;
	default:
		break;
	}
	return data;
}

uint32 exi_read_register32(uint32 addr)
{
	uint32 data;

	data = EXI_READ_UINT32(addr & 0xff);

	switch(addr & 0xff)
	{
	case EXI_REG_CPR0:
		//data |= (1 << 12);
		break;
	case EXI_REG_CPR1:
		//data |= ((1 << 12) | (1 << 7) | (1 << 11) | (1 << 1));
//		data |= (1 << 12) | (1 << 7);
#if !WITH_MEMCARD
		data = 0;
#endif
		break;
	case EXI_REG_CPR2:
		break;
	default:
		break;
	}
	return data;
}

char *exi_tt_string[4] = {"READ", "WRITE", "READ/WRITE", "UNDEF"};

uint32 exi_source_addr;

void exi_run(int ch)
{
	uint32 cpr, cr, len, mar, data;
	cr = EXI_READ_UINT32(EXI_REG_CR0 + (ch * 0x14));
	data = EXI_READ_UINT32(EXI_REG_DATA0 + (ch * 0x14));
	cpr = EXI_READ_UINT32(EXI_REG_CPR0 + (ch * 0x14));
	mar = EXI_READ_UINT32(EXI_REG_MAR0 + (ch * 0x14));
	len = EXI_READ_UINT32(EXI_REG_LEN0 + (ch * 0x14));

	syslog(EXI,"EXI TRANSFER: %s\n", (cr & 0x2) ? "DMA" : "Immediate");
	syslog(EXI,"Channel: %d Device: %d\n", ch, (cpr >> 7) & 0x7);
	syslog(EXI,"Transfer Type: %s\n", exi_tt_string[(cr >> 2) & 0x3]);

	if(cr & 0x1)
	{
		// start transfer
		switch(ch)
		{
		case 0x0:
			if ((cpr >> 7) & 0x2)
			{
				syslog(EXI,"EXI Channel 0 Device 1: Macronix Chipset\n");
				exi_macronix_run();
			}
			else
			{
				syslog(EXI,"EXI Channel 0 Device %d not implemented\n", (cpr >> 7) & 0x7);
			}
			break;
		case 0x1:
			switch((cpr >> 7) & 0x7)
			{
			case 0x1:	// device 0
#if WITH_MEMCARD
				exi_memcard_run(1);
#endif
				break;
			default:
				syslog_error(EXI,"EXI Channel 1 Device %d not implemented\n", (cpr >> 7) & 0x7);
				break;
			}
			break;
		default:
			syslog(EXI,"EXI Channel %d Not Implemented\n", ch);
			break;
		}
	}
	cr &= ~0x1;
	EXI_WRITE_UINT32(EXI_REG_CR0 + (ch * 0x14), cr);
}

/*
	All Macronix devices (SRAM/RTC/ROM/UART/etc) are hooked on DMA ch 0 dev 1
	Addressing mode is always given by first write of immediate data
	bit 29 selects devices other than IPL ROM (if bit 29 == 0 then IPL_ROM requested
	address = DATA0 >> 6
	bit 31 signals writing to device (like SRAM/RTC)

	0x20000000 - RTC
	0x20000100 - SRAM (64 bytes)
	0x20000600 - ??? Bust A Move 3k writes 11 words there (0s and 10th word 0xfe000000)
	0x20001000 - UART
*/

void exi_write_uart(uint32 data, sint8 size)
{
	static FILE *out = NULL;
	if(out == NULL)
		out = fopen("uart.txt", "w");
	size++;
	while (size--)
	{
		fputc(data >> 24, out);
		if ((data >> 24) == 0xd)
			fprintf(stderr, "\n");
		else
			fprintf(stderr, "%c", data >> 24);
		data <<= 8;
	}
	fflush(out);
}

void exi_macronix_run(void)
{
	uint32 cpr, cr, len, mar, data;
	cr = EXI_READ_UINT32(EXI_REG_CR0);
	data = EXI_READ_UINT32(EXI_REG_DATA0);
	cpr = EXI_READ_UINT32(EXI_REG_CPR0);
	mar = EXI_READ_UINT32(EXI_REG_MAR0);
	len = EXI_READ_UINT32(EXI_REG_LEN0);

	if (cr & 0x2)
	{
		// DMA Mode
		syslog(EXI,"Addr: %08x Size: %08x\n", mar, len);
		switch((cr >> 2) & 0x3) // transfer type
		{
		case 0x0:
			if (exi_mx_addr & 0x20000000)
			{
				syslog(EXI,"Not implemented\n");
			}
			else
			{
				// transfer from IPL_ROM
				uint32 saddr;
				unsigned char *ram;
				uint32 i;

				syslog(EXI,"COPY IPL %08x -> RAM %08x size %08x\n", exi_mx_addr >> 6, mar, len);
				saddr = exi_mx_addr >> 6;
				mar &= 0x3fffffff;
				ram = gMemory;
				for(i = 0 ; i < len ; i++)
				{
					ram[mar++] = ipl_rom[saddr++];
				}
				exi_mx_addr = saddr << 6;
			}
			break;
		default:
			syslog_error(EXI,"This mode for DMA from MX not yet implemented\n");
			break;
		}
	}
	else
	{
		// Immediate Mode
		switch((cr >> 2) & 0x3) // transfer type
		{
		case 0x0:
			if(exi_mx_addr == 0x20010000)
			{
				// make UART report itself available
				EXI_WRITE_UINT32(0x10 & 0xff, 0x03000000);
			}
			// read
			syslog(EXI,"READ IMM from MX not yet implemented\n");
			//EXI_WRITE_UINT32(EXI_REG_DATA0, 0);
			//exit(0);
			break;
		case 0x1:
			// write
			// for transfer len == 4 setup exi address
				if (exi_mx_addr == 0)
					exi_mx_addr = data;
				else
					if (exi_mx_addr == 0xa0010000)
						exi_write_uart(data, (cr >> 4) & 0x3);
			break;
		default:
			syslog_error(EXI,"This mode for IMM from MX not yet implemented\n");
			break;
		}
	}
}

uint32 exi_memcard_offset;

extern uint8 mc[];

void mc_irq(void)
{
	uint32	cpr;
	cpr =	EXI_READ_UINT32(EXI_REG_CPR1);
//	cpr |= EXI_EXTINT | EXI_EXIINT;
#if WITH_MEMCARD
	cpr |= 3 << 11;
#else
	cpr |= 1 << 11;
#endif
	EXI_WRITE_UINT32(EXI_REG_CPR1, cpr);
	mc_avail = true;
}
void exi_memcard_run(uint32 card)
{
	uint32	cpr, cr, len, mar, data;
	uint32	datasw;
	
	uint32	base = card * 0x14;

	cr =	EXI_READ_UINT32(EXI_REG_CR0 + (card * 0x14));
	data =	EXI_READ_UINT32(EXI_REG_DATA0 + (card * 0x14));
	cpr =	EXI_READ_UINT32(EXI_REG_CPR0 + (card * 0x14));
	mar =	EXI_READ_UINT32(EXI_REG_MAR0 + (card * 0x14));
	len =	EXI_READ_UINT32(EXI_REG_LEN0 + (card * 0x14));

	uint32	i;
	if (cr & 0x2)
	{
		// DMA Mode
		syslog(EXI,"DMA Mode for Card %d not implemented\n", card);
		switch((cr >> 2) & 0x3) // transfer type
		{
		case 0:
			// READ (memory card -> mem)
			for (i = 0 ; i < len ; i++)
			{
				gMemory[mar + i] = mc[exi_memcard_offset + i];;
				printf("%02x ", mc[exi_memcard_offset + i]);
			}
			break;
		case 1:
			// WRITE (mem -> memory card)
			for (i = 0 ; i < len ; i++)
			{
				printf("%02x ", gMemory[mar + i]);
			}
			break;
		default:
			syslog_error(EXI,"DMA unknown direction %d\n", (cr >> 2) & 0x3);
			break;
		}
		// and the transfer completed interrupt
		cpr |= EXI_TCINT;
		EXI_WRITE_UINT32(EXI_REG_CPR0 + base, cpr);
	}
	else
	{
		// Immediate Mode
		switch((cr >> 2) & 0x3) // transfer type
		{
		case 0x0:
			// read
			syslog(EXI,"MEMCARD read imm: %02x size: %d\n", exi_ch1_cmd[0] & 0xff, (cr >> 4) & 0x3);
			switch(exi_ch1_cmd[0] & 0xff)
			{
			case 0x52:
				// read from memcard
				EXI_WRITE_UINT32(EXI_REG_DATA0 + (card * 0x14), mc_read_data(exi_memcard_offset, (cr >> 4) & 0x3));
				exi_memcard_offset += 4;
				break;
			}
			break;
		case 0x1:
			// write
			datasw = byteswap32(data);
			syslog(EXI,"MEMCARD write imm: (bswapped) %08x.%d\n", data, (cr >> 4) & 0x3);
			if(exi_ch1_cmd_count > 0)
			{
				((uint32 *)exi_ch1_cmd)[exi_ch1_cmd_ptr++] = byteswap32(data);
				exi_ch1_cmd_count--;
				if (exi_ch1_cmd_count == 0)
					mc_execute_cmd();
			}
			else
			{
				exi_ch1_cmd_count = 0;
				exi_ch1_cmd_ptr = 1;
				((uint32 *)exi_ch1_cmd)[0] = byteswap32(data);
				switch(datasw & 0xff)
				{
				case 0x00:
					// get device ID
					syslog(EXI,"MEMCARD Get Device ID\n");
					EXI_WRITE_UINT32(EXI_REG_DATA0 + (card * 0x14), 0x00000080);
					break;
				case 0x83:
					// get card status
					syslog(EXI,"MEMCARD Get Card Status\n");
					EXI_WRITE_UINT32(EXI_REG_DATA0 + (card * 0x14), mc_get_status());
					break;
				case 0x52:
					// read block
					exi_ch1_cmd_count = 2;
					syslog(EXI,"MEMCARD Read Block\n");
					EXI_WRITE_UINT32(EXI_REG_DATA0 + (card * 0x14), 0x00000001);
					exi_memcard_offset = (data & 0x00ffffff) << 1;
					break;
				case 0xf1:
					syslog(EXI,"MEMCARD Erase Sector\n");
					cpr |= EXI_EXIINT;
					EXI_WRITE_UINT32(EXI_REG_CPR0 + base, cpr);
					break;
				case 0xf2:
					exi_ch1_cmd_count = 2;
					syslog(EXI,"MEMCARD Write Block\n");
					cpr |= EXI_EXIINT;
					EXI_WRITE_UINT32(EXI_REG_CPR0 + base, cpr);
					break;
				case 0x81:
					syslog(EXI,"MEMCARD ???????\n");
					break;
				case 0x89:
					// clear card status
					syslog(EXI,"MEMCARD Clear Card Status\n");
					break;
				default:
					syslog_error(EXI,"MEMCARD CMD: %08x not implemented\n", data);
				}
			}
			break;
		default:
			syslog_error(EXI,"MEMCARD unimplemented operation %d\n", (cr >> 2) & 0x3);
			break;
		}
	}
}

bool exi_check_interrupt(void)
{
	uint32	cpr;
	cpr = EXI_READ_UINT32(EXI_REG_CPR1);
	if (cpr & EXI_TCINTMASK)
		if(cpr & EXI_TCINT)
			return true;
	if (cpr & EXI_EXTINTMASK)
		if(cpr & EXI_EXTINT)
			return true;
	if (cpr & EXI_EXIINTMASK)
		if(cpr & EXI_EXIINT)
			return true;
	return false;
}