/*====================================================================

filename:     gdsp_interpreter.cpp
project:      GCemu
created:      2004-6-18
mail:		  duddie@walla.com

Copyright (c) 2005 Duddie & Tratax

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

====================================================================*/
#include <stdio.h>
#include <stdlib.h>
#include "dtypes.h"
#include "gdsp_opcodes.h"
#include "gdsp_interpreter.h"
#include "gdsp_memory.h"
#include "gdsp_registers.h"
#include "gdsp_interface.h"



dsp_t *dsp;


bool (*dsp_op[])(uint16 opc) = 
{
	dsp_op0, dsp_op1, dsp_op2, dsp_op3, 
	dsp_op4, dsp_op5, dsp_op6, dsp_op7, 
	dsp_op8, dsp_op9, dsp_opa, dsp_opb, 
	dsp_opc, dsp_opd, dsp_ope, dsp_opf, 
};

void dbg_error(char *err_msg)
{
	return;
}

void gdsp_init(void)
{
	int i;

	dsp = (dsp_t *)malloc(sizeof(dsp_t));

	dsp->irom = (uint16 *)malloc(DSP_IROM_SIZE * sizeof(uint16));
	dsp->iram = (uint16 *)malloc(DSP_IRAM_SIZE * sizeof(uint16));
	dsp->drom = (uint16 *)malloc(DSP_DROM_SIZE * sizeof(uint16));
	dsp->dram = (uint16 *)malloc(DSP_DRAM_SIZE * sizeof(uint16));

	for(i = 0 ; i < DSP_IRAM_SIZE ; i++)
		dsp->iram[i] = 0x0021; // HALT opcode
	for(i = 0 ; i < DSP_DRAM_SIZE ; i++)
		dsp->dram[i] = 0x0021; // HALT opcode
	dsp->cr = 0x804;
	gdsp_ifx_init();
}

void gdsp_reset(void)
{
	dsp->pc = DSP_RESET_VECTOR;
	dsp->loop = 0;
}

void gdsp_generate_exception(uint8 level)
{
//	printf("Exception level %d at %04x\n", level, dsp->pc);
//	printf("reg $ac0.h %04x\n", dsp->r[0x10]);
	dsp_reg_store_stack(DSP_STACK_C, dsp->pc);
	dsp_reg_store_stack(DSP_STACK_D, dsp->r[0x13]);
	//dsp_reg_store_stack(1, dsp->r[0x10]);
	dsp->pc = level * 2;
}
uint32 gdsp_exception_ready = 0;

void gdsp_exception(uint8 level)
{
	switch(level & 0x7)
	{
	case 0x5:	// ACCA > ACCH exception
		if (dsp->r[0x13] & 0x200)
			gdsp_exception_ready = level;
		break;
	default:
		break;
	}
}


bool gdsp_load_rom(char *fname)
{
	FILE *in;
	in = fopen(fname, "rb");
	printf(fname);
	if (in)
	{
		fread(dsp->irom, 1, DSP_IRAM_SIZE, in);
        fclose(in);
		return true;
	}
	return false;
}



void gdsp_write_cr(uint16 val)
{
	if (val & 0x0001)
		gdsp_reset();
	val &= ~0x0001;
	if ((val & 0x804) == 0x804) 
		dsp->pc = 0x8000;
	if ((val & 0x804) == 0x004) 
		dsp->pc = 0x0000;
	dsp->cr = val;
}

uint16	gdsp_read_cr(void)
{
	if (dsp->pc & 0x8000)
		dsp->cr |= 0x800;
	else
		dsp->cr &= ~0x800;
	return dsp->cr;
}

bool gdsp_step(void)
{
	uint16	opc;
	bool	res;

	dsp->err_pc = dsp->pc;
	opc = dsp_fetch_code();
	res = dsp_op[opc>>12](opc);
	if (dsp->loop)
	{
		if (dsp->pc == (dsp->r[0x0e] + 1))
		{
			dsp->r[0x0f]--;
			if (dsp->r[0x0f])
			{
				dsp->pc = dsp->r[0x0c];
			}
			else
			{
				// end of loop
				dsp->loop = 0;
				dsp_reg_load_stack(0);
				dsp_reg_load_stack(2);
				dsp_reg_load_stack(3);
			}
		}
	}
	if (gdsp_exception_ready)
	{
		gdsp_generate_exception(gdsp_exception_ready);
		gdsp_exception_ready = 0;
	}
	return res;
}
bool gdsp_running;
extern volatile uint32 dsp_running;

bool gdsp_run(void)
{
	gdsp_running = true;
	while(!(dsp->cr & 0x4) && gdsp_running)
	{
		//if (dsp->pc == 0xc50)	break;
		if (!gdsp_step())
		{
			fprintf(stderr, "%04x ERROR: OPCODE %04x NOT IMPLEMENTED\n", dsp->pc, dsp_imem_read(dsp->pc));
			gdsp_running = false;
			exit(0);
		}
	}
	gdsp_running = false;
	return true;
}

bool gdsp_runx(uint16 cnt)
{
	gdsp_running = true;
	while(!(dsp->cr & 0x4) && gdsp_running)
	{
		//if (dsp->pc == 0x027b)	break;
		if (!gdsp_step())
		{
			fprintf(stderr, "%04x ERROR: OPCODE %04x NOT IMPLEMENTED\n", dsp->pc, dsp_imem_read(dsp->pc));
			gdsp_running = false;
			exit(0);
		}
		cnt--;
		if (cnt == 0)
			break;
	}
	gdsp_running = false;
	return true;
}

void gdsp_stop(void)
{
	gdsp_running = false;
}

