#define SWAN_INTERNAL_EEP "swan.eep"
#define SWAN_CONFIG_SRM "swan.srm"

#define BGWIN_CLIP_Y       // BG Window Y clipping enable : LK{!!
#define BUGFIX_NARUTO      // NARUTOBugfix

#define BETWEEN(j,min,max) ((min<=j) && (j<=max))

static FBFORMAT fb_format;

#define VBLANK_START 144 // 145
#define VBLANK_END   158

////////////////////////////////////////////////////////////////////////////////
//
// Wonderswan emulator
// 
////////////////////////////////////////////////////////////////////////////////
#include "hal.h"

#include "ws.h"
#include "nec.h"
#include "necintrf.h"
#include "initval.h"
#include "cstring.h"

#define IO_BACKUP_A4 0xf4
#define IO_BACKUP_A5 0xf5
#define IO_BACKUP_A6 0xf6
#define IO_BACKUP_A7 0xf7

#define CPU_CLOCK          30720000
#define PAL_TILE(tInfo)    (((tInfo)>>9)&0x0f)

#define DRAW_MASK 0x8000

#ifdef WIN32
 static int is0mask = 0x07;
 #define IS_BGBG()     (pWS->ioRam[0x00]&is0mask&0x01)
 #define IS_BGFG()     (pWS->ioRam[0x00]&is0mask&0x02)
 #define IS_SPRT()     (pWS->ioRam[0x00]&is0mask&0x04)
 #define BGCOLORMOD(a) ((a)&0x7fff)
 #define BGBGMOD(a)    (((a))&0x7fff)
 #define BGFGMOD(a)    ((a)&0x7fff)
 #define BGWNMOD(a)    (((a))&0x7fff)
 #define SPMOD(a)      (((a))&0x7fff)
#else
 #define IS_BGBG()     (pWS->ioRam[0x00]&0x01)
 #define IS_BGFG()     (pWS->ioRam[0x00]&0x02)
 #define IS_SPRT()     (pWS->ioRam[0x00]&0x04)
 #define BGCOLORMOD(a) (a)
 #define BGBGMOD(a)    (a)
 #define BGFGMOD(a)    (a)
 #define BGWNMOD(a)    (a)
 #define SPMOD(a)      (a)
#endif

#define BGC_MASK(col)  (BGCOLORMOD(col))
#define BGBGMASK(col)  (BGBGMOD(col))
#define BGFGMASK(col)  (BGFGMOD(col)|DRAW_MASK)
#define BGWNMASK(col)  (BGWNMOD(col)|DRAW_MASK)
#define SPMASK(col)    (SPMOD(col))

typedef struct ws_romHeaderStruct {
    u8	developperId;
    u8	minimumSupportSystem;
    u8	cartId;
    u8  undef_3;
    u8	romSize;
    u8	eepromSize;
    u8	additionnalCapabilities;
    u8	realtimeClock;
    u16	checksum;
} ws_romHeaderStruct;

static int ws_scanline = 0;

// typedef
// WonderSwan Structure : for State Save
typedef struct {
    u8 tag_iram[4];   // State Tag
    u8 iRam[0x10000];
    u8 tag_ioram[4];  // State Tag
    u8 ioRam[0x100];

    u8 tag_spr[4];    // State Tag
    u8 spTableCnt;
    u32 spTable[128];

    u8 videoMode;

    u8 tag_snd[4];    // State Tag
    // Audio Work Parameter
    int noise_k;
    int noise_v;
    int noise_r;
    int noise_c;
    int sweep_pitch;
    int sweep_step;
    int sweep_value;
    int sweep_upd;
    int sweep_count;
    int snd_n[4];
    int snd_k[4];

    // L܂łꊇۑLTCYvZĕۑ
    u8 tag_eE2P[4];    // State Tag
    u8 romE2P[0x800];   // E2P(16kb=2048Byte) in ROM Cart

    u8 tag_eRAM[4];    // State Tag
    u8 romRam[0x40000]; // RAM( 2Mb=256KByte) in ROM Cart

} WONDERSWAN_T;

// TILE CACHE (temporary buffer)
typedef struct {
    u32 pal_flag;
    u8 wsc_n[0x10000];
    u8 wsc_h[0x10000];
    u8 modify[1024];
    
    PIXEL_FORMAT ws_shades[16];
    PIXEL_FORMAT system_c_pal[256]; // ˑp̃pbge[u
    PIXEL_FORMAT system_m_pal[8];   // ˑp̃pbge[u
    
} SYSTEM_CACHE_T;

// internal functions
//static u16* ws_gpu_palette(int pal_idx);

static void ws_reset(void);
void ws_set_colour_scheme(int scheme);
static void ws_gpu_changeVideoMode(u8 value);
static void ws_gpu_clearCache(void);
static void ws_memory_init(u8 *rom, u32 wsRomSize);

// internal values
static WONDERSWAN_T* pWS=0;
static SYSTEM_CACHE_T* pWC=0;
static BYTE* ws_RomAddr = 0;
static DWORD ws_RomSize = 0;

// export functions
void cpu_writeport(DWORD port,BYTE value);
void cpu_writemem20(DWORD addr,BYTE value);
static void render_update_sound(int st_ch,int ed_ch,int do_cursor);

static void ws_audio_dma(void);


// export values
static u32 ws_key;
int rtcDataRegisterReadCount=0;
u8* pWsRomMap[0x10] = { 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0 };

//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
#define IO_ROM_BANK_BASE_SELECTOR	0xC0

// e|̈
static u32 sramAddressMask;
static u32 romAddressMask;
static u32 eEepromAddressMask;


#define SOUND_SamplesPerFrame 588 //735 //588 // 44100/75=588
#define SWAN_SOUND_CHANNELS   4
#define WAVE_SAMPLES          32

static int waveTable[SWAN_SOUND_CHANNELS][WAVE_SAMPLES];

#define VOLUME_R(ch)     (pWS->ioRam[0x88+ch]&15)
#define VOLUME_L(ch)    ((pWS->ioRam[0x88+ch]>>4)&15)
#define GET_PITCH(ch)   ((((int)pWS->ioRam[0x81+(ch)*2]&7)<<8)|((int)pWS->ioRam[0x80+(ch)*2]))



//------------------------------------------------------------------------------
//-
//------------------------------------------------------------------------------
static void ws_reset(void)
{
    int i;

    ws_key = 0;
    core_memset(pWS->iRam,0,sizeof(pWS->iRam));

   	/* Set Default Scheme */
	ws_set_colour_scheme(0);  

    pWS->spTableCnt=0;
    
    for (i=0;i<0x0c9;i++) {
        cpu_writeport(i,initialIoValue[i]);
    }
    
    rtcDataRegisterReadCount=0;
    
    /* ws_gpu_init */
    ws_gpu_clearCache();
    ws_gpu_changeVideoMode(2);

    nec_reset(NULL);
	nec_set_reg(NEC_SP,0x2000);
}


//------------------------------------------------------------------------------
//- EmulatorRA1/75^C~OŌĂяo֐
//------------------------------------------------------------------------------
static void RefreshSound(void)
{
    if( HAL_Sound() ) {
        render_update_sound(0,4,SOUND_SamplesPerFrame);
        HAL_Sound_Proc32(halSnd.R32,halSnd.L32,SOUND_SamplesPerFrame);
        halSnd.ch[0]=halSnd.ch[1]=halSnd.ch[2]=halSnd.ch[3]=0;
    }
}

//------------------------------------------------------------------------------
//-
//------------------------------------------------------------------------------
void ws_set_colour_scheme(int scheme)
{
    int r,g,b,i;
    int R,G,B;

    if(pWS) {
        switch(scheme) {
          default:
          case 0: r=100; g=100; b=100;  break;// white
          case 1: r= 60; g= 61; b=  0;  break;// amber
          case 2: r= 20; g= 90; b= 20;  break;// green
          case 3: r=  0; g= 61; b= 60;  break;// blue
        }
        
        for(i=0;i<16;i++) {
            R = (((15-i) * r)/100);
            G = (((15-i) * g)/100);
            B = (((15-i) * b)/100);
            pWC->ws_shades[i] = HAL_fb2_Color(R,G,B,RGB444);
            //pWC->ws_shades[i]&= 0x7ffe;
        }
    }
}


//------------------------------------------------------------------------------
//- rfI[ḧɖ肪B
//- 0 : F4,5,6,7pbgɑ, I/Opbg𗘗p
//- 4 : F4,5,6,7pbgɑ, MEMpbg𗘗p
//- 6 : F͑Spbgɑ, 7Ƃ̓^C̐@Ⴄ
//- 7 : F͑Spbgɑ, 6Ƃ̓^C̐@Ⴄ
//------------------------------------------------------------------------------
static void ws_gpu_changeVideoMode(u8 value)
{
    value = (value>>5) & 7;

    switch(value) {
      case 7:  pWS->videoMode=7; break;
      case 6:  pWS->videoMode=6; break;
      case 4:  pWS->videoMode=4; break;
      default: pWS->videoMode=0; break;
	}

	// cache clearȂ̂SAGAŃ^CgĂ
	ws_gpu_clearCache();
}

//------------------------------------------------------------------------------
// ŃpbgƃoOB
// e[YQŕ\ȂȂ
// core_memset(pWC,0,sizeof(*pWC));
// ƂĂ܂̂
//------------------------------------------------------------------------------
static void ws_gpu_clearCache(void)
{
    pWC->pal_flag = -1;
	core_memset(pWC->wsc_n,0,sizeof(*pWC->wsc_n));
	core_memset(pWC->wsc_h,0,sizeof(*pWC->wsc_h));
	core_memset(pWC->modify,-1,sizeof(pWC->modify));
}



//------------------------------------------------------------------------------
//- Get Palette Pointer (COLOR / MONO)
//------------------------------------------------------------------------------
static PIXEL_FORMAT* ws_gpu_palette(int pal_idx)
{
    // 0 : PaletteXV
    if( (pWC->pal_flag & (1<<pal_idx)) ) {

		int p = 0x20 + pal_idx * 2;

        pWC->pal_flag &= ~(1<<pal_idx);
        
        if(pWS->videoMode==0) {
			pWC->system_c_pal[pal_idx*16 + 0] = pWC->system_m_pal[ (pWS->ioRam[p+0]   )&7 ];
			pWC->system_c_pal[pal_idx*16 + 1] = pWC->system_m_pal[ (pWS->ioRam[p+0]>>4)&7 ];
			pWC->system_c_pal[pal_idx*16 + 2] = pWC->system_m_pal[ (pWS->ioRam[p+1]   )&7 ];
            pWC->system_c_pal[pal_idx*16 + 3] = pWC->system_m_pal[ (pWS->ioRam[p+1]>>4)&7 ];
        }
    }

    return &pWC->system_c_pal[pal_idx*16];
}

//------------------------------------------------------------------------------
//-
//------------------------------------------------------------------------------
static u8* ws_tileCache_getTileRow(u32 tileIndex, u32 line,
                                      u32 vFlip, u32 hFlip, 
                                      u32 bank)
{
#define PHN(h,n) pH[(h)] = pN[(n)]

    int i;
    u32 tL,tLP;
    
    if(pWS->videoMode>=4 && bank){
        tileIndex+=512;
    }
    
    if(pWC->modify[tileIndex]) {
        u8* pN = &pWC->wsc_n[tileIndex<<6];
        u8* pH = &pWC->wsc_h[tileIndex<<6];

        pWC->modify[tileIndex]=0;

		switch( pWS->videoMode ) {
		case 7: { // 1pixel = 4bit
            u32 *tIRP = (u32*)&pWS->iRam[0x4000+(tileIndex<<5)];
			for(i=0;i<8;i++) {
                tL=*tIRP++;
                PHN(7,0) = (tL>> 4)&0x0f;
                PHN(6,1) = (tL    )&0x0f;
                PHN(5,2) = (tL>>12)&0x0f;
                PHN(4,3) = (tL>> 8)&0x0f;
                PHN(3,4) = (tL>>20)&0x0f;
                PHN(2,5) = (tL>>16)&0x0f;
                PHN(1,6) = (tL>>28)     ;
                PHN(0,7) = (tL>>24)&0x0f;
                pN+=8;
                pH+=8;
            }
		} break;

		case 6: { // 1pixel = 4bit
            u32 *tIRP = (u32*)&pWS->iRam[0x4000+(tileIndex<<5)];
			for(i=0;i<8;i++) {
                tL=*tIRP++;
                tLP = (tL>>7)&0x01010101; PHN(7,0) = tLP|(tLP>>7)|(tLP>>14)|(tLP>>21);
                tLP = (tL>>6)&0x01010101; PHN(6,1) = tLP|(tLP>>7)|(tLP>>14)|(tLP>>21);
                tLP = (tL>>5)&0x01010101; PHN(5,2) = tLP|(tLP>>7)|(tLP>>14)|(tLP>>21);
                tLP = (tL>>4)&0x01010101; PHN(4,3) = tLP|(tLP>>7)|(tLP>>14)|(tLP>>21);
                tLP = (tL>>3)&0x01010101; PHN(3,4) = tLP|(tLP>>7)|(tLP>>14)|(tLP>>21);
                tLP = (tL>>2)&0x01010101; PHN(2,5) = tLP|(tLP>>7)|(tLP>>14)|(tLP>>21);
                tLP = (tL>>1)&0x01010101; PHN(1,6) = tLP|(tLP>>7)|(tLP>>14)|(tLP>>21);
                tLP = (tL>>0)&0x01010101; PHN(0,7) = tLP|(tLP>>7)|(tLP>>14)|(tLP>>21);
                pN+=8;
                pH+=8;
            }				
		} break;

		default: {  // 1pixel = 2bit
			u16 *tIRP = (u16*)&pWS->iRam[0x2000+(tileIndex<<4)];
			for(i=0;i<8;i++) {
				tL=*tIRP++;
                tLP = (tL>>7)&0x0101; PHN(7,0) = tLP|(tLP>>7);
	            tLP = (tL>>6)&0x0101; PHN(6,1) = tLP|(tLP>>7);
		        tLP = (tL>>5)&0x0101; PHN(5,2) = tLP|(tLP>>7);
			    tLP = (tL>>4)&0x0101; PHN(4,3) = tLP|(tLP>>7);
				tLP = (tL>>3)&0x0101; PHN(3,4) = tLP|(tLP>>7);
				tLP = (tL>>2)&0x0101; PHN(2,5) = tLP|(tLP>>7);
				tLP = (tL>>1)&0x0101; PHN(1,6) = tLP|(tLP>>7);
				tLP = (tL>>0)&0x0101; PHN(0,7) = tLP|(tLP>>7);
				pN+=8;
				pH+=8;
			}
		}break;
		}

        // tile cache updated
    }

    if (vFlip) line=7-line;
    if (hFlip) return (&pWC->wsc_h[(tileIndex<<6)+(line<<3)]);
    else       return (&pWC->wsc_n[(tileIndex<<6)+(line<<3)]);

    return(NULL);
}


//------------------------------------------------------------------------------
//- ^C\(obNOEh)
//- ԍŏɕ`悷̂œBGBGFœhׂKvB
//- x^h肵ĂBG`悷蓧F=BGFƏBG`sB
//------------------------------------------------------------------------------
static void DrawScreen_BackBg(PIXEL_FORMAT* framebuffer ,PIXEL_FORMAT bgc)
{
    int pixels;
    int scrX = pWS->ioRam[0x10];
    int scrY =(pWS->ioRam[0x11]+ws_scanline)&0xff;
    int	lineInTile = scrY&0x07;
    int cit = scrX&0x07;
    int ws_currentTile = (scrX>>3);

    u32 bgbgBase = ((u32)(pWS->ioRam[7] & 0x0f)) << 11;
    PIXEL_FORMAT* ws_bgScrollRamBase = (u16*)(pWS->iRam+bgbgBase+((scrY&0xfff8)<<3));
    
    // Optimize Target
    for(pixels=224;pixels;) {
        u16	tInfo= ws_bgScrollRamBase[ws_currentTile&0x1f];
        u8*  wsTR = ws_tileCache_getTileRow(tInfo&0x1ff, lineInTile, tInfo&0x8000, tInfo&0x4000, tInfo&0x2000);
        PIXEL_FORMAT* wsPA = ws_gpu_palette(PAL_TILE(tInfo));
        PIXEL_FORMAT  pa0 = wsPA[0];
        int max = (pixels<8)?pixels:8-cit;
        
        pixels-=max;
        wsTR+=cit;

        // KvȂ瓧FBGFɈꎞIɕύX
        if((pWS->videoMode>4) || (tInfo&0x800)) { wsPA[0]=bgc; }

        switch(max) {
          case 8: *framebuffer++ = BGBGMASK(wsPA[*wsTR++]) ;
          case 7: *framebuffer++ = BGBGMASK(wsPA[*wsTR++]);
          case 6: *framebuffer++ = BGBGMASK(wsPA[*wsTR++]);
          case 5: *framebuffer++ = BGBGMASK(wsPA[*wsTR++]);
          case 4: *framebuffer++ = BGBGMASK(wsPA[*wsTR++]);
          case 3: *framebuffer++ = BGBGMASK(wsPA[*wsTR++]);
          case 2: *framebuffer++ = BGBGMASK(wsPA[*wsTR++]);
          case 1: *framebuffer++ = BGBGMASK(wsPA[*wsTR++]);
          default: break;
        }
        
        // F𕜌
        wsPA[0] = pa0;
        cit=0;
        ws_currentTile++;
    }
}

//------------------------------------------------------------------------------
//- 
//- Foreground TILE draw (with window clip(inside/outside))
//- 
//------------------------------------------------------------------------------
static void DrawScreen_BackFg(PIXEL_FORMAT* framebuffer)
{
    s32 pixels;
    s32 windowMode=pWS->ioRam[0x00]&0x30;
    s32 scrX=pWS->ioRam[0x12];
    s32 scrY=((s32)pWS->ioRam[0x13]+ws_scanline)&0xff;
	s32 cT=(scrX>>3);
	u16* ws_fgScrollRamBase;
    s32	lit = scrY&0x07; // LineInTile
    s32 cit = scrX&0x07;

    s32 side = (windowMode>>4)&1; // I/Oݒ肪O(=1)(=0)H

#ifdef BGWIN_CLIP_Y  // CLIPYĂ݂
    u32 bgWin_y0=pWS->ioRam[0x09];
    u32 bgWin_y1=pWS->ioRam[0x0b];
	int yclip = 0;
#endif/*BGWIN_CLIP_Y*/

    u32 bgfgBase = ((u32)(pWS->ioRam[7] & 0xf0)) <<  7;
    
    ws_fgScrollRamBase=(u16*)(pWS->iRam+bgfgBase+ ((scrY&0xfff8)<<3));

	// Window Mode EnablȅꍇłWindowsvȂ̂WindowȂP 
	if( windowMode & 0x20 ) {
		if(bgWin_y0<=ws_scanline && ws_scanline<=bgWin_y1) { 
			yclip=1; 
		} else {
			if(side==1) {
				windowMode = 0;
			}
		}
	}

    // window disabled
    if((windowMode&0x20)==0) {

//		if(ws_scanline<100 && ws_scanline<120) { return; }

        for(pixels=224;pixels>0;) {
            u16 tInfo= ws_fgScrollRamBase[cT&0x1f];
            u8* wsTR = ws_tileCache_getTileRow( tInfo&0x1ff, lit,tInfo&0x8000, tInfo&0x4000, tInfo&0x2000);
            PIXEL_FORMAT* wsPA = ws_gpu_palette(PAL_TILE(tInfo));

			pixels-=(8-cit);
            wsTR+=cit;

            if((pWS->videoMode>4) || (tInfo&0x800)) {
                for(;cit<8;cit++,wsTR++,framebuffer++) {
                    if(*wsTR) {
                        *framebuffer = BGFGMASK(wsPA[*wsTR]);
					}
					// else *framebuffer = 0x1f<<10;
                }
            } else {
                for(;cit<8;cit++,wsTR++,framebuffer++) {
                    *framebuffer = BGFGMASK(wsPA[*wsTR]);
                }
            }
            cit=0;
            cT++;
        }
    }
    else { // foreground layer displayed with Window
        int bgWin_x0=pWS->ioRam[0x08];
        int bgWin_x1=pWS->ioRam[0x0a];
        int column=0;

        for(pixels=224;pixels>0;) {
            u16	tInfo= ws_fgScrollRamBase[cT&0x1f];
            u8*  wsTR = ws_tileCache_getTileRow(tInfo&0x1ff, lit,tInfo&0x8000, tInfo&0x4000, tInfo&0x2000);
            PIXEL_FORMAT* wsPA = ws_gpu_palette(PAL_TILE(tInfo));

			pixels-=(8-cit);
            wsTR+= cit;

            if((pWS->videoMode>4) || (tInfo&0x800)) { 
                for(;cit<8;cit++,wsTR++,framebuffer++,column++) {
					int ans = yclip && BETWEEN(column,bgWin_x0,bgWin_x1); //  && bgWin_x0<=column && column<=bgWin_x1;

                    if( (side^ans) && *wsTR ) {
                        *framebuffer = BGWNMASK(wsPA[*wsTR]);
                    }
                }
            } else {
                for(;cit<8;cit++,wsTR++,framebuffer++,column++) {
					int ans = yclip && BETWEEN(column,bgWin_x0,bgWin_x1); // yclip && bgWin_x0<=column && column<=bgWin_x1;

					if( side ^ ans ) *framebuffer = BGWNMASK(wsPA[*wsTR]);
                }
            }
            
            cit=0;
            cT++;
		}
    }
}

//------------------------------------------------------------------------------
//- XvCg\
//- \ɊւăoO݂悤B
//- <e>
//- XvCgm̕\ԂӎĂȂ̂ŏB
//- \ԂSP0...SP127ŁASP`掞SPBGȂBG̃`FbN
//- EBhE݂Ȃ = hׂ֎~][݂Ȃ
//------------------------------------------------------------------------------
static void DrawScreen_Sprite(PIXEL_FORMAT* framebuffer)
{
    int i,j;
    int spWin_x0 = pWS->ioRam[0x0c];
    int spWin_y0 = pWS->ioRam[0x0d];
    int spWin_x1 = pWS->ioRam[0x0e];
    int spWin_y1 = pWS->ioRam[0x0f];
	byte spWin   = pWS->ioRam[0x00] & 0x08; // Sprite Window Enable?
	u8 * wsTR;
	PIXEL_FORMAT* wsPA,*fb;
    u32* pSpRam,spr,t,p;
	s32 x,y;
	int outside=0,top_pri=0,xbet,ybet;

	if( spWin_x0==spWin_x1) {
		spWin &= ~0x08;
	}

	// EBhEȂ = (0,0)-(223,143)̓`̃EBhE
	if(!spWin) {
		spWin_x0 = 0;
		spWin_y0 = 0;
		spWin_x1 = 224;
		spWin_y1 = 144;
		outside  = 0; // inside
	}

	/* YW͈͓tO */
	ybet = BETWEEN(ws_scanline,spWin_y0,spWin_y1);

    pSpRam = &pWS->spTable[pWS->spTableCnt-1];

    for(i=0;i<pWS->spTableCnt;i++) {
		spr = *pSpRam--;
		x = (spr>>24) & 0x0ff;
		y = (spr>>16) & 0x0ff;
		t =  spr      & 0x1ff;
		p =((spr&0xe00)>>9) + 8;

		// cygnełĂ钲Ă݂
		if(y>(144-1+7)) {
			y = (signed char)((unsigned char)y);
		}

		if((y+8)<=(s32)ws_scanline) continue;
		if( y   > (s32)ws_scanline) continue;

		if(x>=(224-1+7)) {
			x = (signed char)((unsigned char)x);
		}

		if(x<=-8) continue;
		if((y>=144)||(x>=224))  continue;

		top_pri = (spr & 0x2000)<<2;
		outside = (spr&0x1000)&&(spWin);

		fb = framebuffer;
    
		wsTR = ws_tileCache_getTileRow(t,(ws_scanline-y)&0x07,(spr&0x8000),(spr&0x4000),0);
		wsPA = ws_gpu_palette(p);

		if((pWS->videoMode>4) || (p&0x04)) {
			// (\)
			// if(!outside &&  BETWEEN) true 
			// if(outside  && !BETWEEN) true 
			// ƂXORg
			for(j=x;j<(x+8);j++,wsTR++) {
				if( (fb[j]&DRAW_MASK) && !top_pri ) { continue; }
					
				if(*wsTR) {
					xbet = BETWEEN(j,spWin_x0,spWin_x1);
					if( outside ^ (xbet && ybet) ) {
						fb[j]= (fb[j]&DRAW_MASK) | SPMASK(wsPA[*wsTR]);
					}
#ifdef WIN32
					else {
						//fb[j]=0x1f<<5;// fb[j]; for debug
					}
#endif
				}
//				wsTR++;
			}
		}
		else {
			for(j=x;j<(x+8);j++,wsTR++) {
				if( (fb[j]&DRAW_MASK) && !top_pri ) continue;
				xbet = BETWEEN(j,spWin_x0,spWin_x1);
				if( outside ^ (xbet && ybet) ) {
					fb[j] = SPMASK(wsPA[*wsTR]);
				}
#ifdef WIN32
				else {
					//fb[j]=0x1f<<5;// fb[j]; for debug
				}
#endif
//				wsTR++;
			}
		}
	}
}

//------------------------------------------------------------------------------
//-
//------------------------------------------------------------------------------
BYTE cpu_readport(BYTE port)
{
    switch(port) {
      case 0x92: /* Noise Counter Shift Register */
	  case 0x93: /* `lSLȂKɕω */
		  if(pWS->ioRam[0x90]&0x08) { // near random value
			  pWS->ioRam[port] += ((port&1)*2) + 1;
		  }
		  break;

      case 0x00: case 0x01: case 0x02: case 0x03: case 0x04: case 0x05: case 0x06: case 0x07:
      case 0x08: case 0x09: case 0x0A: case 0x0B: case 0x0C: case 0x0D: case 0x0E: case 0x0F:
      case 0x10: case 0x11: case 0x12: case 0x13: case 0x14: case 0x15: case 0x16: case 0x17:
      case 0x18: case 0x19: case 0x1A: case 0x1B: case 0x1C: case 0x1D: case 0x1E: case 0x1F:
      case 0x20: case 0x21: case 0x22: case 0x23: case 0x24: case 0x25: case 0x26: case 0x27:
      case 0x28: case 0x29: case 0x2A: case 0x2B: case 0x2C: case 0x2D: case 0x2E: case 0x2F:
      case 0x30: case 0x31: case 0x32: case 0x33: case 0x34: case 0x35: case 0x36: case 0x37:
      case 0x38: case 0x39: case 0x3A: case 0x3B: case 0x3C: case 0x3D: case 0x3E: case 0x3F:
      case 0x40: case 0x41: case 0x42: case 0x43: case 0x44: case 0x45: case 0x46: case 0x47:
      case 0x48: case 0x49: case 0x4A: case 0x4B: case 0x4C: case 0x4D: case 0x4E: case 0x4F:
      case 0x50: case 0x51: case 0x52: case 0x53: case 0x54: case 0x55: case 0x56: case 0x57:
      case 0x58: case 0x59: case 0x5A: case 0x5B: case 0x5C: case 0x5D: case 0x5E: case 0x5F:
      case 0x60: case 0x61: case 0x62: case 0x63: case 0x64: case 0x65: case 0x66: case 0x67:
      case 0x68: case 0x69: case 0x6A: case 0x6B: case 0x6C: case 0x6D: case 0x6E: case 0x6F:
      case 0x70: case 0x71: case 0x72: case 0x73: case 0x74: case 0x75: case 0x76: case 0x77:
      case 0x78: case 0x79: case 0x7A: case 0x7B: case 0x7C: case 0x7D: case 0x7E: case 0x7F:
      case 0x80: case 0x81: case 0x82: case 0x83: case 0x84: case 0x85: case 0x86: case 0x87:
      case 0x88: case 0x89: case 0x8A: case 0x8B: case 0x8C: case 0x8D: case 0x8E: case 0x8F:
      case 0x90: case 0x91: case 0x94: case 0x95: case 0x96: case 0x97:
      case 0x98: case 0x99: case 0x9A: case 0x9B: case 0x9C: case 0x9D: case 0x9E: case 0x9F: break;
        
      case 0xA0: return 0x87;
      case 0xA1: case 0xA2: case 0xA3: case 0xA4: case 0xA5: case 0xA6: case 0xA7: case 0xA8: case 0xA9: break;
      case 0xAA: return 0xff;
      case 0xAB: case 0xAC: case 0xAD: case 0xAE: case 0xAF: case 0xB0: case 0xB1: case 0xB2: break;
      case 0xB3: 
        if (pWS->ioRam[0xb3]<0x80) return 0;
        if (pWS->ioRam[0xb3]<0xc0) return 0x84;
        return 0xc4;
        
      case 0xB4:
      case 0xB5: /* Controler */
      case 0xB6:
      case 0xB7:
      case 0xB8:
      case 0xB9: break;
      case 0xBA:{/* eeprom even byte read */
          int w1=((((u16)pWS->ioRam[0xbd])<<8)|((u16)pWS->ioRam[0xbc]));
          w1=(w1<<1)&0x3ff;
          return internalEeprom[w1];
      }
        
      case 0xBB:{/* eeprom odd byte read */
          int w1=((((u16)pWS->ioRam[0xbd])<<8)|((u16)pWS->ioRam[0xbc]));
          w1=((w1<<1)+1)&0x3ff;
          return internalEeprom[w1];
      }
        
      case 0xBC:
      case 0xBD:
      case 0xBE:
      case 0xBF:// break;
      case 0xC0:
//        return ((pWS->ioRam[0xc0]&0xf)|0x20);
        
      case 0xC1:
      case 0xC2:
      case 0xC3: break;
        
      case 0xC4:
        if(eEepromAddressMask) {
            int w1=(((((WORD)pWS->ioRam[0xc7])<<8)|((WORD)pWS->ioRam[0xc6]))<<1)&(eEepromAddressMask);
            return pWS->romE2P[w1];
        }
#ifdef WIN32
        *(int*)0xc4 = -1;
#endif
        return 0xff;
        
      case 0xC5:
        if(eEepromAddressMask) {
            int w1=(((((WORD)pWS->ioRam[0xc7])<<8)|((WORD)pWS->ioRam[0xc6]))<<1)&(eEepromAddressMask);
            return pWS->romE2P[w1+1];
        }
#ifdef WIN32
        *(int*)0xc4 = -1;
#endif
        
      case 0xC6:
      case 0xC7: break;
        
      case 0xC8:
        if(eEepromAddressMask) {
            // ack eeprom write
            if(pWS->ioRam[0xc8]&0x20)
              return pWS->ioRam[0xc8]|2;
            
            // ack eeprom read
            if(pWS->ioRam[0xc8]&0x10)
              return pWS->ioRam[0xc8]|1;

            // else ack both
            return pWS->ioRam[0xc8]|3;
        }
#ifdef WIN32
        *(int*)0xc8 = -1;
#endif

      case 0xC9:
		  break;

      case 0xCA:
		return pWS->ioRam[0xca] | 0x80;
		  break;
        
      case 0xCB:  // RTC data register
		if( pWS->ioRam[0xca]==0x15 ) {
			switch(rtcDataRegisterReadCount) {
			case 0: rtcDataRegisterReadCount++; return 0; /* BCD : year + 2000 */
			case 1: rtcDataRegisterReadCount++; return 0; /* BCD : month       */
			case 2: rtcDataRegisterReadCount++; return 0; /* BCD : day         */
			case 3: rtcDataRegisterReadCount++; return 0; /* BCD : day of week */
			case 4: rtcDataRegisterReadCount++; return 0; /* BCD : hour        */
			case 5: rtcDataRegisterReadCount++; return 0; /* BCD : min         */
			case 6: rtcDataRegisterReadCount=0; return 0; /* BCD : sec         */
			}
		} else {
            return (pWS->ioRam[0xcb]|0x80);
		}
        break;
        
      case 0xCC:
      case 0xCD:
      case 0xCE:
      case 0xCF:
      case 0xD0: case 0xD1: case 0xD2: case 0xD3: case 0xD4: case 0xD5: case 0xD6: case 0xD7:
      case 0xD8: case 0xD9: case 0xDA: case 0xDB: case 0xDC: case 0xDD: case 0xDE: case 0xDF:
      case 0xE0: case 0xE1: case 0xE2: case 0xE3: case 0xE4: case 0xE5: case 0xE6: case 0xE7:
      case 0xE8: case 0xE9: case 0xEA: case 0xEB: case 0xEC: case 0xED: case 0xEE: case 0xEF:
        break;

      case 0xF0: case 0xF1: case 0xF2: case 0xF3: case 0xF4: case 0xF5: case 0xF6: case 0xF7:
      case 0xF8: case 0xF9: case 0xFA: case 0xFB: case 0xFC: case 0xFD: case 0xFE: case 0xFF:
        return 0xd1; // F0-FFƗpƂĎgp
		  break;
    }
    return pWS->ioRam[port];
}

// 
// 
// 
void cpu_writeport(DWORD port,BYTE value)
{
    int w1,ix;

    /* 背WX^lŉ쐬 */
    switch(port){
      case 0x52: // Audio DMA
      case 0x90:
      case 0x94: // Main Volume Control
        render_update_sound(0,4,((int)ws_scanline*3722)/1000);
        break;
      case 0x80:
      case 0x81:
      case 0x88:
        render_update_sound(0,1,((int)ws_scanline*3722)/1000);
        break;
      case 0x82:
      case 0x83:
      case 0x89:
        render_update_sound(1,2,((int)ws_scanline*3722)/1000);
        break;
      case 0x8c: /* Audio 3 Sweep value */
      case 0x8d: /* Audio 3 Sweep step  */
        pWS->sweep_upd = 1;
      case 0x84:
      case 0x85:
      case 0x8a:
        render_update_sound(2,3,((int)ws_scanline*3722)/1000);
        break;
      case 0x86:
      case 0x87:
      case 0x8b:
      case 0x8e: /* Audio 4 Noise Control */
        render_update_sound(3,4,((int)ws_scanline*3722)/1000);
        break;

      case 0x4e: /* DMA address <read only> */
      case 0x4f: /* DMA address <read only> */
        break;

      case 0xb6:
        if(value) {
            pWS->ioRam[0xb6] &= ~value;
        }
        return;
    }

#ifdef WIN32
    {   // port access debug
        void swan_port_debug(int after,byte port,byte value);
        swan_port_debug(0,port,value);
        pWS->ioRam[port]=value;
        swan_port_debug(1,port,value);
    }
#else
    pWS->ioRam[port]=value;
#endif

    switch (port) {
      case 0x00: // Display Control
      case 0x01: // Background Color
      case 0x02: // Current Line
      case 0x03: // Line Compare (for Interrupt)
      case 0x04: //
      case 0x05:
      case 0x06:
#ifdef WIN32
        port = port;
#endif
        break;
      case 0x07:
//        pWS->bgbgBase = ((u32)(pWS->ioRam[7] & 0x0f)) << 11;
//        pWS->bgfgBase = ((u32)(pWS->ioRam[7] & 0xf0)) <<  7;
        break;

      case 0x08:
      case 0x09:
      case 0x0A:
      case 0x0B:
      case 0x0C:
      case 0x0D:
      case 0x0E:
      case 0x0F:
      case 0x10:
      case 0x11:
		  break;

      case 0x12:
      case 0x13:
#ifdef WIN32
		  value=value;
#endif
		  break;

      case 0xa4: 
	  case 0xa5: // HBLANK
      case 0xa6: 
	  case 0xa7: // VBLANK
        pWS->ioRam[port|0xf0]=value; // backup to unused area
        break;
        
      case 0xa8:
      case 0xa9:
        break;

      case 0x1c:
      case 0x1d:
      case 0x1e:
      case 0x1f:
        pWC->system_m_pal[(port-0x1c)*2+0] = pWC->ws_shades[15 & value];
        pWC->system_m_pal[(port-0x1c)*2+1] = pWC->ws_shades[(value>>4)];
        pWC->pal_flag = -1;
        break;
        
      case 0x20:  case 0x21:  case 0x22:  case 0x23:
      case 0x24:  case 0x25:  case 0x26:  case 0x27:
      case 0x28:  case 0x29:  case 0x2a:  case 0x2b:
      case 0x2c:  case 0x2d:  case 0x2e:  case 0x2f:
      case 0x30:  case 0x31:  case 0x32:  case 0x33:
      case 0x34:  case 0x35:  case 0x36:  case 0x37:
      case 0x38:  case 0x39:  case 0x3a:  case 0x3b:
      case 0x3c:  case 0x3d:  case 0x3e:  case 0x3f:
        //        pWS->ws_pal[((port&0x1f)*2)+0] = 7 & value;
        //        pWS->ws_pal[((port&0x1f)*2)+1] = 7 & (value>>4);
        pWC->pal_flag |= (1<<((port-0x20)/2));
        break;

      case 0x42:
      case 0x43:
        pWS->ioRam[port] &= 0x0f;
        break;

      case 0x48:	// DMA
        // bit 7 set to start dma transfer
        if(value&0x80) {
            int dma_start = (((DWORD)pWS->ioRam[0x41])<<8)|(((DWORD)pWS->ioRam[0x40]))|(((DWORD)pWS->ioRam[0x42])<<16);
            int dma_end   = (((DWORD)pWS->ioRam[0x45])<<8)|(((DWORD)pWS->ioRam[0x44]))|(((DWORD)pWS->ioRam[0x43])<<16);
            int dma_size  = (((DWORD)pWS->ioRam[0x47])<<8)|(((DWORD)pWS->ioRam[0x46]));

#ifdef  BUGFIX_NARUTO
            dma_end &= 0x000fffff; // naruto bugfix
#endif//BUGFIX_NARUTO

            for(ix=0;ix<dma_size;ix++) {
                cpu_writemem20(dma_end++,cpu_readmem20(dma_start++));
            }

            pWS->ioRam[0x47]=0;
            pWS->ioRam[0x46]=0;
            pWS->ioRam[0x41]=(BYTE)(dma_start>>8);
            pWS->ioRam[0x40]=(BYTE)(dma_start&0xff);
            pWS->ioRam[0x45]=(BYTE)(dma_end>>8);
            pWS->ioRam[0x44]=(BYTE)(dma_end&0xff);
            pWS->ioRam[0x48]=0;
        }
        break;

      case 0x4a: /* sound DMA source address */
      case 0x4b:
        break;

      case 0x4c: /* DMA source memory segment bank */
#ifdef WIN32
        if(pWS->ioRam[0x4c]>0x0f) {
            *(int*)0 = 0x4c;
        }
        pWS->ioRam[0x4c]&=0x0f;
#endif
          break;
        
        
      case 0x4d:
      case 0x4e: /* DMA Transfer size (in bytes) */
      case 0x4f: /* ^^^                          */
        break;
      case 0x50: if(value) *(int*)0x50 = value; break;
      case 0x51: if(value) *(int*)0x51 = value; break;

      case 0x52: /* bit  7 = 1  -> DMA start */
		ws_audio_dma();
        break;

      case 0x60:
        ws_gpu_changeVideoMode(value);
        return;

      case 0x80: /* Audio 1 Freq : low  */
      case 0x81: /* ^^^          : high */
      case 0x82: /* Audio 2 freq : low  */
      case 0x83: /* ^^^          : high */
      case 0x84: /* Audio 3 Freq : low  */
      case 0x85: /* ^^^          : high */
      case 0x86: /* Audio 4 freq : low  */
      case 0x87: /* ^^^          : high */
      case 0x88: /* Audio 1 volume      */
      case 0x89: /* Audio 2 volume      */
      case 0x8a: /* Audio 3 volume      */
      case 0x8b: /* Audio 4 volume      */
      case 0x8c: /* Audio 3 Sweep Value */
      case 0x8d: /* Audio 3 Sweep Step  */
        break;

      case 0x8e: /* Audio 4 Noise Control*/
        // Counter Enable
/*        if((value&0x14)==0x14)*/ {
            switch(value&0x7) {
              case 0: pWS->noise_c=(value&7)*512; break;
              case 1: pWS->noise_c=(value&7)*512; break;
              case 2: pWS->noise_c=(value&7)*512; break;
              case 3: pWS->noise_c=(value&7)*512; break;
              case 4: pWS->noise_c=(value&7)*512; break;
              case 5: pWS->noise_c=(value&7)*512; break;
              case 6: pWS->noise_c=(value&7)*512; break;
              case 7: pWS->noise_c=(value&7)*512; break;
            }
		}
		// FF1 Noise fix
		if(value&0x10){
            pWS->noise_k = 0;
            pWS->noise_r = 0;
            pWS->noise_v = 0x51f631e4;
        }
        break;

      case 0x8f: /* Sample Location */
#ifdef WIN32
		  value = value;
#endif
        break;

      case 0x90: /* Audio Control   */
        break;
      case 0x91: /* Audio Output    */
        pWS->ioRam[0x91]|= 0x80; // stereo status
        break;

      case 0x92: /* Noise Counter Shift Register : low?  */
      case 0x93: /* ^^^                          : high? */
//			value = value;
		  break;

      case 0x94: /* Volume 4bit */
        //        ws_audio_port_write(port,value);
        break;

      case 0xb5: // Controls
        switch(value&0xf0) {
          case 0x10: value=0x10 | (0x0f&(ws_key>>8)); break; // read vertical
          case 0x20: value=0x20 | (0x0f&(ws_key>>4)); break; // read horizontal
          case 0x40: value=0x40 | (0x0f&(ws_key)   ); break; // read buttons
          default:   value&=0xf0;                     break;
        }
        pWS->ioRam[0xb5] = value;

        break;

      case 0xba:
        w1=(((((WORD)pWS->ioRam[0xbd])<<8)|((WORD)pWS->ioRam[0xbc])));
        w1=(w1<<1)&0x3ff;
        internalEeprom[w1]=value;
        return;

      case 0xbb:
        w1=(((((WORD)pWS->ioRam[0xbd])<<8)|((WORD)pWS->ioRam[0xbc])));
        w1=((w1<<1)+1)&0x3ff;
        internalEeprom[w1]=value;
        return;

      case 0xbe: // EEPROM
        if(value & 0x20) { value |= 0x02; }
        else if(value & 0x10) { value |= 0x01; }
        else { value|=0x03; }
        pWS->ioRam[0xbe] = value;
        break;


      case 0xc0: {
          int romBank,bank;

          for(bank=4;bank<16;bank++) {
              romBank=(256-(((value&0xf)<<4)|(bank&0xf)));
              pWsRomMap[bank+0x00] = &ws_RomAddr[(unsigned)(ws_RomSize-(romBank<<16))];
          }

          // Read PortŕK̂Write PortɂĂ܂
          pWS->ioRam[0xc0]=(pWS->ioRam[0xc0]&0x0f)|0x20;
      } break;

      case 0xc1: // SRAM Bank Change
		  switch(sramAddressMask) {
		    case 0x1ffff: pWsRomMap[0x01] = &pWS->romRam[(value&1)<<16]; break; 
		    case 0x3ffff: pWsRomMap[0x01] = &pWS->romRam[(value&3)<<16]; break; 
		    default:      pWsRomMap[0x01] = &pWS->romRam[0];             break;
		  }
		  break;
        
      case 0xc2:
        pWsRomMap[0x02] = &ws_RomAddr[ ((value&((ws_RomSize>>16)-1))<<16) ];
        break;

      case 0xc3:
        pWsRomMap[0x03] = &ws_RomAddr[ ((value&((ws_RomSize>>16)-1))<<16) ];
        break;

      case 0xc4:
        w1=(((((WORD)pWS->ioRam[0xc7])<<8)|((WORD)pWS->ioRam[0xc6]))<<1)&eEepromAddressMask;
        pWS->romE2P[w1]=value;
        return;

      case 0xc5:
        w1=(((((WORD)pWS->ioRam[0xc7])<<8)|((WORD)pWS->ioRam[0xc6]))<<1)&eEepromAddressMask;
        pWS->romE2P[w1+1]=value;
        return;

      case 0xca:
        if(value==0x15) {
            rtcDataRegisterReadCount=0;
        }
        break;

	  case 0xf4:
	  case 0xf5:
	  case 0xf6:
	  case 0xf7:
#ifdef WIN32
		value = value;
#endif
        break;

      default:
        break;
    }
}

void cpu_writeport2(DWORD port,WORD value)
{
	cpu_writeport(port  ,value);
	cpu_writeport(port+1,value>>8);
}

// 
// 
// 
void cpu_writemem20(DWORD addr,BYTE value)
{
	u32 bank   = addr>>16;
    u32 offset = addr & 0x0ffff;

    /* 0 - RAM - 16 KB (WS) / 64 KB (WSC) internal RAM */
	if(bank==0) {
        if(pWS->iRam[offset]!=value) {
            pWS->iRam[offset] = value;

            if(offset<0x02000) { /* 0x0000 - 0x1FFF */                                              } else 
			if(offset<0x04000) { /* 0x2000 - 0x3FFF */ pWC->modify[(offset>>4)&0x1ff]=1;       } else 
			if(offset<0x08000) { /* 0x4000 - 0x7FFF */ pWC->modify[(offset>>5)&0x1ff]=1;       } else 
			if(offset<0x0C000) { /* 0x8000 - 0xBFFF */ pWC->modify[((offset>>5)&0x1ff)+512]=1; } else 
			if(offset<0x0FE00) { /* 0xC000 - 0xFDFF */                                              } else {
                int adr = offset & 0x0fffe;
                int pal = (adr - 0xfe00)/2;
                u16 c = ((u16)(pWS->iRam[adr+1]<<8) | pWS->iRam[adr]);
                pWC->system_c_pal[pal] = HAL_fb2_Color((c>>8)&0xf,(c>>4)&0xf,(c&0xf),RGB444);
            }
        }
    } else if(bank==1){
        *(pWsRomMap[0x01]+offset) = value;
    }
#ifdef WIN32
    else {
        /* Ӗȏ݂ĂȂH */
        addr = addr;
    }
#endif
    
	// other banks are read-only
}


//------------------------------------------------------------------------------
// Update WaveTable 
//------------------------------------------------------------------------------
static void wave_update(int ch)
{
    int i;
    byte * wTbl = &pWS->iRam[ (((int)pWS->ioRam[0x8f])<<6) ]; // wave table
    
    for(i=0;i<WAVE_SAMPLES/2;i++) {
        waveTable[ch][i*2+0] = (((int)( wTbl[i+16*ch]&15 ))-8); // 0 to 15
        waveTable[ch][i*2+1] = (((int)((wTbl[i+16*ch]>>4)))-8); // 0 to 15
    }
}

static int psgtbl[15] = { 0, 120, 240, 360, 480, 600, 720, 840, 960, 1080, 1200, 1320, 1440, 1560, 1680 };
                        

//------------------------------------------------------------------------------
// PSG and SWEEP mixer
//------------------------------------------------------------------------------
static void mixer_psg_sweep(int ch,int do_cursor,int freq)
{
    if(freq>0) {
        int volR = psgtbl[ VOLUME_R(ch) ]; //VOLUME_R(ch)
        int volL = psgtbl[ VOLUME_L(ch) ]; //VOLUME_L(ch)
        DWORD na = pWS->snd_n[ch];// & 31;
        DWORD ka = pWS->snd_k[ch];// & 31;
        DWORD NTp,t,i;
        int wavena;
        
        // G[`FbN
        wave_update(ch);
        
        NTp = CPU_CLOCK / freq;
        
        for(i=halSnd.ch[ch];i<do_cursor;i++) {
            wavena = waveTable[ch][na];
            halSnd.R32[i] += (wavena * volR);//*120;
            halSnd.L32[i] += (wavena * volL);//*120;
            ka += NTp;

            // t2ȏɂȂꍇ̂ŒPɏĂ͂Ȃ
            t = ka / (441000);
            na = (na+t)%(WAVE_SAMPLES);
            ka -= 441000*t;
        }
        
        pWS->snd_n[ch]=na;
        pWS->snd_k[ch]=ka;
    }
}

//------------------------------------------------------------------------------
// PSG Mixer (CH1,2,3,4)
//------------------------------------------------------------------------------
static void mixer_psg(int ch,int do_cursor)
{
    mixer_psg_sweep(ch,do_cursor,2048-GET_PITCH(ch));
}

//------------------------------------------------------------------------------
// Voice Mixer (CH2 ONLY)
//------------------------------------------------------------------------------
static void mixer_voice(int ch,int do_cursor)
{
    int voice = pWS->ioRam[0x89]; // 0to255
    int i;

#if 1
    voice = (voice-127) * 128;
#else
	if( voice < 64 ) voice = 64; else
	if( voice >191 ) voice = 192;
	voice = (voice - 128) * 255;
#endif

    for(i=halSnd.ch[ch];i<do_cursor;i++) {
        halSnd.R32[i] += voice;
        halSnd.L32[i] += voice;
    }
}


//------------------------------------------------------------------------------
// Sweep Mixer (CH3 ONLY)
//------------------------------------------------------------------------------
// - channel 3 - sweep - two parameters:
// - step = 2.667 x (N + 1) ms , where N = 5 bit value (0 to 31)
// - value - signed byte (-128 to 127)
//
// 1000(ms) / 75     = 13.333 (ms)
// 1000(ms) / 75 * 2 = 26.67  (ms)
//
// (N+1)=10 (2.667*10)=26.67 : 150t[łP񌸎Z
// (N+1)=20 (2.667*20)=53.34 : 300t[łP񌸎Z
// ā[Ƃ
// (N+1)= 1 (2.667* 1)= 2.67 :  15t[łP񌸎Z
// (N+1)=32 (2.667*32)=85.34 : 480t[łP񌸎Z
// ܂
// SWEEP͐ݒl15Z^C~OŃXEB[v
//------------------------------------------------------------------------------
static void mixer_sweep(int ch,int do_cursor)
{
    if(pWS->sweep_upd) {
        pWS->sweep_pitch = GET_PITCH(ch);
        pWS->sweep_value = (char)pWS->ioRam[0x8c];
        pWS->sweep_step  = pWS->ioRam[0x8d]&31;
        pWS->sweep_upd   = 0;
        pWS->sweep_count = (pWS->sweep_step+1) * 15;
    } else {
        if(do_cursor==SOUND_SamplesPerFrame) {
            if(--pWS->sweep_count==0) {
                pWS->sweep_count = (pWS->sweep_step+1) * 15;
                pWS->sweep_pitch += pWS->sweep_value;

                if(pWS->sweep_pitch<0){
                    pWS->sweep_pitch=0;
                }
                else {
                    if(pWS->sweep_pitch>2047) {
                        pWS->sweep_pitch=2047;
                    }
                }
				pWS->ioRam[0x90] &= ~0x40;
            }
        }
    }
    
    mixer_psg_sweep(ch,do_cursor,2048 - pWS->sweep_pitch);
}

static int noitbl[15] = { 0, 120, 240, 360, 480, 600, 720, 840, 960, 1080, 1200, 1320, 1440, 1560, 1680 };


//------------------------------------------------------------------------------
//  NOISE Mixer (CH4 ONLY)
//------------------------------------------------------------------------------
static void mixer_noise(int ch,int do_cursor)
{
    int i;
    int volR = noitbl[ VOLUME_R(ch) ]; // VOLUME_R(ch)
    int volL = noitbl[ VOLUME_L(ch) ]; // VOLUME_L(ch)
    
    for(i=halSnd.ch[ch];i<do_cursor;i++) {
        pWS->noise_k += 3000 + pWS->noise_c;
        
        if(pWS->noise_k>=44100) {
            if(pWS->noise_v & 0x80000) {
                pWS->noise_v = ((pWS->noise_v ^ 0x04)<<1)+1;
                pWS->noise_r = 1;
            } else {
                pWS->noise_v <<= 1;
                pWS->noise_r = 0;
            }
            
            pWS->noise_k -= 44100;
        }
        
        halSnd.L32[i] += ((pWS->noise_r?(10):(-10))*volL);
        halSnd.R32[i] += ((pWS->noise_r?(10):(-10))*volR);
    }
}


//------------------------------------------------------------------------------
// 
//------------------------------------------------------------------------------
static void render_update_sound(int st_ch,int ed_ch,int do_cursor)
{
    if( HAL_Sound() ) {
        void (*pMixerFunc)(int ch,int do_cursor);
        int ch;
        
        for(ch=st_ch;ch<ed_ch;ch++) {
            pMixerFunc = mixer_psg;
            switch(ch) {
              case 0: // CH1 : Audio
                if(!(pWS->ioRam[0x90]&0x01))    pMixerFunc = 0; // silent
                break;
              case 1: // CH2 : Audio + Voice
                if(!(pWS->ioRam[0x90]&0x02))    pMixerFunc = 0; else
                if((pWS->ioRam[0x90]&0x20))     pMixerFunc = mixer_voice; // silent
                break;
              case 2: // CH3 : Audio + Sweep
                if(!(pWS->ioRam[0x90]&0x04))    pMixerFunc = 0; else
                if( (pWS->ioRam[0x90]&0x40))    pMixerFunc = mixer_sweep; 
                break;
              case 3: // CH4 : Audio + Noise
                if(!(pWS->ioRam[0x90]&0x08))    pMixerFunc = 0; else // silent
                if( (pWS->ioRam[0x90]&0x80))    pMixerFunc = mixer_noise;
                break;
              default:
                pMixerFunc = 0; // silent
            }

            if(pMixerFunc) {
                (pMixerFunc)(ch,do_cursor);
            }

            halSnd.ch[ch]=do_cursor;
        }
    }
}

//------------------------------------------------------------------------------
// 
//------------------------------------------------------------------------------
static int ws_save(int fd)
{
    int size,len=0;
    void* pCpu = nec_getRegPtr(&len);

    memcpy(pWS->tag_iram, "*RAM",4);
    memcpy(pWS->tag_ioram,"*I/O",4);
    memcpy(pWS->tag_spr,  "*SPR",4);
    memcpy(pWS->tag_snd,  "*SND",4);
    memcpy(pWS->tag_eE2P, "@E2P",4);
    memcpy(pWS->tag_eRAM, "@RAM",4);

    size = sizeof(*pWS)
         - sizeof(pWS->romE2P) - sizeof(pWS->tag_eE2P) 
         - sizeof(pWS->romRam) - sizeof(pWS->tag_eRAM);

    HAL_sts_write(fd,pCpu,len);
    HAL_sts_write(fd,pWS,size);
    
    if(sramAddressMask) {
        HAL_sts_write(fd,pWS->tag_eRAM,sramAddressMask+1 + sizeof(pWS->tag_eRAM) );
    }
    else if(eEepromAddressMask) {
        HAL_sts_write(fd,pWS->tag_eE2P,eEepromAddressMask+1 + sizeof(pWS->tag_eE2P));
    }
    
    return 1;
}

//------------------------------------------------------------------------------
// 
//------------------------------------------------------------------------------
static int ws_load(int fd)
{
    int i,size,len=0;
    void *pCpu = nec_getRegPtr(&len);
    byte buf[5];

    size = sizeof(*pWS)
         - sizeof(pWS->romE2P) - sizeof(pWS->tag_eE2P) 
         - sizeof(pWS->romRam) - sizeof(pWS->tag_eRAM);

    HAL_sts_read(fd,pCpu,len);
    HAL_sts_read(fd,pWS,size);

    if(sramAddressMask)    {
        HAL_sts_read(fd,pWS->tag_eRAM,sramAddressMask+1);
    }
    else if(eEepromAddressMask) {
        HAL_sts_read(fd,pWS->tag_eE2P,eEepromAddressMask+1);
    }
    
    // Tile Cache Clear
    ws_gpu_clearCache();
    core_memset(waveTable,0,sizeof(waveTable));
    
    // ROM Map Update
    cpu_writeport(0xc0,pWS->ioRam[0xc0]);
    cpu_writeport(0xc2,pWS->ioRam[0xc2]);
    cpu_writeport(0xc3,pWS->ioRam[0xc3]);

    // Update Color Palette
    if( pWS->videoMode>4 ) {
        byte v;

        for(i=0xfe00;i<0x10000;i++) {
            v = cpu_readmem20(i);
            cpu_writemem20(i,0);
            cpu_writemem20(i,v);
        }
    } else {
        for(i=0x1c;i<=0x1f;i++) {
            cpu_writeport(i,pWS->ioRam[i]);
        }
    }

    return 1;
}

//------------------------------------------------------------------------------
// 
//------------------------------------------------------------------------------
void ws_int_check(void)
{
	int intr = pWS->ioRam[0xb6] & pWS->ioRam[0xb2];
	byte mask=8;

	if(intr) {
#if 1
		if(intr&0x20) mask=5; else // VBLANK end
		if(intr&0x40) mask=6; else // VBLANK begin
		if(intr&0x80) mask=7; else // HBLANK Timer
		if(intr&0x10) mask=4; else // SCANLINE
#else
		if(intr&0x80) mask=7; else // HBLANK Timer
		if(intr&0x40) mask=6; else // VBLANK begin
		if(intr&0x20) mask=5; else // VBLANK end
		if(intr&0x10) mask=4; else // SCANLINE
#endif
			return ;

/*      ͈̔͂͋IOFFɂĂ̂ł肦Ȃ
		if(intr&0x08) mask=3; else
		if(intr&0x04) mask=2; else
		if(intr&0x02) mask=1; else 
		if(intr&0x01) mask=0; 
*/
		
		if(mask<8) {
			if( nec_int((pWS->ioRam[0xb0]+mask)*4) ) {
//				pWS->ioRam[0xb6]&=~(1<<mask);
			}
		}
	}
}


//------------------------------------------------------------------------------
// Audio DMA work
//------------------------------------------------------------------------------
static void ws_audio_dma(void)
{
	if(pWS->ioRam[0x52]&0x80) {
		int adr = (int)((int)pWS->ioRam[0x4c]<<16) | ((int)pWS->ioRam[0x4b]<<8) | pWS->ioRam[0x4a];
		int len = (int)((int)pWS->ioRam[0x4f]<<8 ) | ((int)pWS->ioRam[0x4e]);

        pWS->ioRam[0x90] |= 0x22;
        
		cpu_writeport(0x89,cpu_readmem20(adr));

		adr++;
		len--;

        /* Sound DMAbank𒴂鎖Β~ */
        if( (adr>>16) != pWS->ioRam[0x4c] ) {
#ifdef WIN32
            *(int*)0 = 0xDADADADA;
#endif
            adr--;
            len=0;
        }

#if 1 /* oswan */
        if(len<32) {
            len=0;
        }
#endif
        
//      pWS->ioRam[0x4C]=(u8)((adr>>16)&0xFF); /* Bankz͋֎~ */
		pWS->ioRam[0x4B]=(u8)((adr>>8)&0xFF);
		pWS->ioRam[0x4A]=(u8)(adr&0xFF);
		pWS->ioRam[0x4F]=(u8)((len>>8)&0xFF);
		pWS->ioRam[0x4E]=(u8)(len&0xFF);

		if(len==0) {
			pWS->ioRam[0x52]&=~0x80;
#if 0 // Terrors2ŉłȂ̏C
            pWS->ioRam[0x90]&=~0x22;
#endif
		}
	}
}

static void drawLine(int bDrawLine)
{
	// Scanline Video Processing
    if(bDrawLine) {
		PIXEL_FORMAT *fb,bg;

        if(ws_scanline<VBLANK_START /*144*/) {
            fb = fb_format.fb + (fb_format.width * ws_scanline);
                
            if(pWS->videoMode) { bg = pWC->system_c_pal[pWS->ioRam[1]];   }
            else               { bg = pWC->system_m_pal[pWS->ioRam[1]&7]; }
                
            if(IS_BGBG())      { DrawScreen_BackBg(fb,BGC_MASK(bg)); }
            else               { int i; for(i=0;i<224;i++) fb[i]=bg; }

			if(IS_BGFG()) DrawScreen_BackFg(fb);
            if(IS_SPRT()) DrawScreen_Sprite(fb);
		}
	}
}

static void update_sprite_table(void)
{
	// XvCge[u͎IɕύX(pbg͕)
    if( ws_scanline==VBLANK_START/*144*/ ) {
		u32 base = (((u32)pWS->ioRam[4]&0x3f)<<9) + pWS->ioRam[5]*4; 
        pWS->spTableCnt = pWS->ioRam[6];
        core_memcpyX4(pWS->spTable,(u32*)&pWS->iRam[base],pWS->spTableCnt*4);
	}
}


//------------------------------------------------------------------------------
// 
//------------------------------------------------------------------------------
static void ws_executeLine(PIXEL_FORMAT *framebuffer, int renderLine)
{	
    for(ws_scanline=0;ws_scanline<159;ws_scanline++) {

//        ws_audio_dma();
        
        pWS->ioRam[2]=ws_scanline;

// <Interrupt Flag Update>
        if(pWS->ioRam[0xa2]&0x01) {
#if 0
			int hblank_timer;
			hblank_timer = pWS->ioRam[0xa5];
			hblank_timer = (hblank_timer<<8) | pWS->ioRam[0xa4];
			    
			if(hblank_timer && (pWS->ioRam[0xa2]&0x01)) { /*HBLANK COUNTUP*/
				hblank_timer--;
				pWS->ioRam[0xa4]=hblank_timer&0xff;
				pWS->ioRam[0xa5]=(hblank_timer>>8)&0xff;
				if(!hblank_timer&&(pWS->ioRam[0xb2]&0x80)) {
					pWS->ioRam[0xb6]|=(pWS->ioRam[0xb2]&0x80);
				}
				if(!hblank_timer&&(pWS->ioRam[0xa2]&0x02)) {
					pWS->ioRam[0xa4] = pWS->ioRam[IO_BACKUP_A4];
					pWS->ioRam[0xa5] = pWS->ioRam[IO_BACKUP_A5];
				}
			}
#else
			if(pWS->ioRam[0xa4]||pWS->ioRam[0xa5]) {
				if( pWS->ioRam[0xa4]==0 ) { 
					pWS->ioRam[0xa5]--;    // 0xa5͐΂0ȊO
	                pWS->ioRam[0xa4]=255;
		        } else {
			        if((--pWS->ioRam[0xa4])==0) {
				        if(pWS->ioRam[0xa5]==0) {
					        pWS->ioRam[0xb6]|=(pWS->ioRam[0xb2]&0x80);
						    if(pWS->ioRam[0xa2]&0x02) {
							    pWS->ioRam[0xa4] = pWS->ioRam[IO_BACKUP_A4];
								pWS->ioRam[0xa5] = pWS->ioRam[IO_BACKUP_A5];
	                        }
		                }
			        }
				}
			}
#endif
		}
        
        if(ws_scanline==VBLANK_START) {   // VBLANK begin INT
            pWS->ioRam[0xb6] |= pWS->ioRam[0xb2]&0x40;
#if 1 // test
            if(!(++pWS->ioRam[0xaa])) {
                if(!(++pWS->ioRam[0xab])) {
                    if(!(++pWS->ioRam[0xac])) {
                        ++pWS->ioRam[0xad];
                    }
                }
            }
#endif
        } 

        else if(ws_scanline==VBLANK_END) {
            if((pWS->ioRam[0xa2]&0x04)) {
				// JE^[ZbgĂꍇ̓JEg
				if( pWS->ioRam[0xa6]|| pWS->ioRam[0xa7]) {
				    if( pWS->ioRam[0xa6]==0 ) {
                        pWS->ioRam[0xa7]--;
                        pWS->ioRam[0xa6]=255;
					} else {
                        if((--pWS->ioRam[0xa6])==0) {
						    if(pWS->ioRam[0xa7]==0) {
							    pWS->ioRam[0xb6]|=(pWS->ioRam[0xb2]&0x20);
								if(pWS->ioRam[0xa2]&0x80) {
									pWS->ioRam[0xa6] = pWS->ioRam[IO_BACKUP_A6];
									pWS->ioRam[0xa7] = pWS->ioRam[IO_BACKUP_A7];
					            }
					        }
					    }
					}
				} 
				// JE^[ZbgĂȂꍇ̓JEgɊ荞
				else {
					pWS->ioRam[0xb6]|=(pWS->ioRam[0xb2]&0x20);
				}
            }
        }

#if 1
        // 0x10 : SCANLINE INT(Drawing line timer)
        if(ws_scanline==pWS->ioRam[0x3]) {
            pWS->ioRam[0xb6] |= pWS->ioRam[0xb2]&0x10; 
        }
#endif

#if 1
        pWS->ioRam[0xb6] &= ~0x0f;
#else
        if(pWS->ioRam[0xb2]&0x08) { pWS->ioRam[0xb6]&=~0x08; } // 0x08 : Serial recieve  (incomplete)
        if(pWS->ioRam[0xb2]&0x04) { pWS->ioRam[0xb6]&=~0x04; } // 0x04 : RTC Alarm       (incomplete)
        if(pWS->ioRam[0xb2]&0x02) { pWS->ioRam[0xb6]&=~0x02; } // 0x02 : Key press       (incomplete)
        if(pWS->ioRam[0xb2]&0x01) { pWS->ioRam[0xb6]&=~0x01; } // 0x01 : Serial transmit (incomplete)
#endif
// </Interrupt Flag Update>

        ws_audio_dma();

		if(0) {
			ws_int_check();
		    nec_execute(256);
			drawLine(renderLine);
		}
		else {
	        ws_int_check();
			drawLine(renderLine);
		    nec_execute(256);
		}

		update_sprite_table();
	}

    RefreshSound();
}


//
// ŝqnl]~Ă邩
//
int isWonderSwanRotate(void)
{
    if(ws_RomAddr) {
        return ws_RomAddr[ws_RomSize-4]&1;
    }

    return 0;
}

static int SWAN_Reset(void)
{
    if(pWS && pWC) {
        ws_reset();
    }
}

/*
//------------------------------------------------------------------------------
// MEMORY BANK MAP
// BANK 0 : INTERNAL RAM
// BANK 1 : SRAM (Cart inside)
// BANK 2 : 
//------------------------------------------------------------------------------
static void ws_memory_init(u8 *rom, u32 wsRomSize)
{
    ws_romHeaderStruct* pRomHeader = (ws_romHeaderStruct*)&rom[wsRomSize-10];

    core_memset(pWsRomMap,0,sizeof(pWsRomMap));
    pWsRomMap[0] = pWS->iRam;
    pWsRomMap[1] = pWS->romRam;

    sramAddressMask = eEepromAddressMask=0;
    
    // SRAM
    switch (pRomHeader->eepromSize) {
      case 0x01: sramAddressMask    = 0x01fff; break; //  64kbit =  8KB (SRAM)
      case 0x02: sramAddressMask    = 0x07fff; break; //  256kbit= 32KB 
      case 0x03: sramAddressMask    = 0x1ffff; break; // 1024kbit=128KB
      case 0x04: sramAddressMask    = 0x3ffff; break; // 2048kbit=256KB
      case 0x10: eEepromAddressMask = 0x0007f; break; //    1kbit= 128B
      case 0x20: eEepromAddressMask = 0x007ff; break; //   16kbit=2048B
	  case 0x50: eEepromAddressMask = 0x000ff; break; //    8kbit=1024B
      case 0x00:
      default:
        break;
    }
    romAddressMask = wsRomSize-1;
}
*/

//------------------------------------------------------------------------------
// ROM[hꂽɈxs֐
// ̊֐łROMɊւ鏈Lq邵A
// ZbgɕKvȏ͕ʊ֐Ɏ邱
//------------------------------------------------------------------------------
static int ws_init(char *pRom,int nRom)
{
    ws_romHeaderStruct* pRomHeader = 0;

	core_memset(pWS,0,sizeof(WONDERSWAN_T));

    core_memset(pWsRomMap,0,sizeof(pWsRomMap));
    pWsRomMap[0] = pWS->iRam;
    pWsRomMap[1] = pWS->romRam;

    sramAddressMask = eEepromAddressMask=0;

	{
		ws_romHeaderStruct* pRomHeader = (ws_romHeaderStruct*)&pRom[nRom-10];
    
		// SRAM
		switch (pRomHeader->eepromSize) {
		  case 0x01: sramAddressMask    = 0x01fff; break; //  64kbit =  8KB (SRAM)
		  case 0x02: sramAddressMask    = 0x07fff; break; //  256kbit= 32KB 
		  case 0x03: sramAddressMask    = 0x1ffff; break; // 1024kbit=128KB
		  case 0x04: sramAddressMask    = 0x3ffff; break; // 2048kbit=256KB
		  case 0x10: eEepromAddressMask = 0x0007f; break; //    1kbit= 128B
		  case 0x20: eEepromAddressMask = 0x007ff; break; //   16kbit=2048B
		  case 0x50: eEepromAddressMask = 0x000ff; break; //    8kbit=1024B
		  case 0x00:
		  default:
			break;
		}
    
		romAddressMask = nRom-1;
	}

#if 1 /* ws_patchRom(pRom,nRom) */
    if((pRom[nRom-10]==0x01)&&(pRom[nRom-8]==0x27)) { // Detective Conan 
        // WS cpu is using cache/pipeline or
        //   there's protected ROM bank where 
        //   pointing CS 
        pRom[0xfffe8]=(char)0xea;
		pRom[0xfffe9]=(char)0x00;
		pRom[0xfffea]=(char)0x00;
		pRom[0xfffeb]=(char)0x00;
		pRom[0xfffec]=(char)0x20;
		
	}
#endif
    
    return 1;
}

//------------------------------------------------------------------------------
//-
//------------------------------------------------------------------------------
static int SWAN_Init(int nRomSize,byte* pRomAddr)
{
    ws_RomAddr = pRomAddr;
    ws_RomSize = nRomSize;
    ws_key = 0;

    if( (pWS = (WONDERSWAN_T*)HAL_mem_malloc(sizeof(WONDERSWAN_T))) && 
        (pWC = (SYSTEM_CACHE_T*)HAL_mem_malloc(sizeof(SYSTEM_CACHE_T))) ) {
            
        if(ws_init(pRomAddr,nRomSize)) {

			ws_reset();
            
            HAL_fb2_init(256,256,&fb_format,HW_WSC);
            fb_format.pic_w = 224;
            fb_format.pic_h = 144;
            
            // READ : EEPROM(Internal)
            HAL_Cfg_Load(SWAN_INTERNAL_EEP,internalEeprom,sizeof(internalEeprom));
            
            // ROM [SRAM or EEPROM]
            if(sramAddressMask) {
                HAL_Mem_Load(pWS->romRam,sramAddressMask+1);
            }
            else if(eEepromAddressMask) {
                HAL_Mem_Load(pWS->romE2P,eEepromAddressMask+1);
            }
            
            return 1;
        }
    }

    return 0;
}

static int SWAN_Loop(void)
{
    int f_draw = !HAL_fps(75);
    
    ws_executeLine(fb_format.fb,f_draw);
    
    if(f_draw) {
        HAL_fb2_bitblt(&fb_format);
    }

    return (ws_key=HAL_Input(0,HW_WSC)) & (1<<31);
}


static int SWAN_Exit(void)
{
    // WRITE : EEPROM(Internal)
    HAL_Cfg_Save(SWAN_INTERNAL_EEP,internalEeprom,sizeof(internalEeprom));
    
	// ROM [SRAM or EEPROM]
    if(sramAddressMask) {
        HAL_Mem_Save(pWS->romRam,sramAddressMask+1);
    }
    else if(eEepromAddressMask) {
        HAL_Mem_Save(pWS->romE2P,eEepromAddressMask+1);
    }

    ws_RomAddr = 0;
    ws_RomSize = 0;

    if(pWS) HAL_mem_free(pWS); 
    if(pWC) HAL_mem_free(pWC);

    pWS = 0;
    pWC=0;
    
    return 1;
}

int SWAN_Setup(void)
{
    HAL_SetupExt(EXT_WS ,"ws" ,
                 SWAN_Init,SWAN_Loop,SWAN_Exit,SWAN_Reset,
                 ws_load, ws_save
                 );

    HAL_SetupExt(EXT_WSC,"wsc",
                 SWAN_Init,SWAN_Loop,SWAN_Exit,SWAN_Reset,
                 ws_load, ws_save
                 );

    return 1;
}


